<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\ProfileController;
use App\Http\Controllers\WalletTransactionController;
use App\Http\Controllers\TradeController;
use App\Http\Controllers\Admin\AdminTradeController;
use Inertia\Inertia;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
*/

// ------------------------
// Home / Landing Page
// ------------------------
use App\Http\Controllers\MarketController;

Route::get('/market-data', [MarketController::class, 'getData']);

Route::get('/', function () {
    return Inertia::render('Home', [
        'auth' => ['user' => auth()->user()],
    ]);
})->name('home');

// ------------------------
// Dashboard Route (Redirect based on role)
// ------------------------
Route::middleware(['auth', 'verified'])->get('/dashboard', function () {
    $user = auth()->user();

    if ($user->role === 1) {
        return redirect()->route('admin.home');
    } else {
        return redirect()->route('user.dashboard');
    }
})->name('dashboard');

// ------------------------
// Admin Routes
// ------------------------


Route::middleware(['auth', 'verified', 'admin'])->group(function () {
    // Admin Dashboard
    Route::get('/admin-home', function () {
        return Inertia::render('Admin/AdminHome', [
            'auth' => ['user' => auth()->user()],
            'users' => \App\Models\User::latest()->get(),
            'trades' => \App\Models\Trade::with('user')->latest()->take(5)->get(),
            'transactions' => \App\Models\WalletTransaction::with('user')->latest()->take(5)->get(),
        ]);      
    })->name('admin.home');

    // Admin Wallet Page
    Route::get('/admin-wallet', [WalletTransactionController::class, 'index'])->name('admin.wallet');
    Route::post('/admin-wallet', [WalletTransactionController::class, 'adminStore'])->name('admin.wallet.store');
    Route::put('/admin-wallet/{id}', [WalletTransactionController::class, 'adminUpdate'])->name('admin.wallet.update');
    Route::delete('/admin-wallet/{id}', [WalletTransactionController::class, 'adminDestroy'])->name('admin.wallet.destroy');

    // Approve / Decline Transactions
    Route::post('/admin-wallet/{id}/approve', [WalletTransactionController::class, 'approve'])->name('admin.wallet.approve');
    Route::post('/admin-wallet/{id}/decline', [WalletTransactionController::class, 'decline'])->name('admin.wallet.decline');

    // Approve / Decline Trades
    Route::get('/admin-trades', [AdminTradeController::class, 'index'])->name('admin.trades');
    Route::get('/admin-trades-pending', [AdminTradeController::class, 'index']);
    // Approve / Decline
    Route::post('/admin-trade/{id}/approve', [AdminTradeController::class, 'approve'])->name('admin.trade.approve');
    Route::post('/admin-trade/{id}/decline', [AdminTradeController::class, 'decline'])->name('admin.trade.decline');
    // CRUD Operations
    Route::post('/admin-trade', [AdminTradeController::class, 'store'])->name('admin.trade.store');
    Route::put('/admin-trade/{id}', [AdminTradeController::class, 'update'])->name('admin.trade.update');
    Route::delete('/admin-trade/{id}', [AdminTradeController::class, 'destroy'])->name('admin.trade.destroy');

    // Admin Support Chat
    Route::get('/admin-support', [App\Http\Controllers\ChatController::class, 'index'])->name('admin.support');
    Route::get('/api/admin/chats', [App\Http\Controllers\ChatController::class, 'getAllChats'])->name('admin.api.chats');
    Route::get('/api/admin/chat/{id}/messages', [App\Http\Controllers\ChatController::class, 'getChatMessages'])->name('admin.api.chat.messages');
    Route::post('/api/admin/chat/{id}/reply', [App\Http\Controllers\ChatController::class, 'adminReply'])->name('admin.api.chat.reply');

    // KYC Management
    Route::get('/admin-kyc', [App\Http\Controllers\AdminKycController::class, 'index'])->name('admin.kyc');
    Route::get('/api/admin/kyc-requests', [App\Http\Controllers\AdminKycController::class, 'getRequests'])->name('admin.api.kyc');
    Route::post('/api/admin/kyc/{id}/update', [App\Http\Controllers\AdminKycController::class, 'updateStatus'])->name('admin.api.kyc.update');

    // Chart Manipulation
    Route::get('/admin-charts', [App\Http\Controllers\AdminChartController::class, 'index'])->name('admin.charts');
    Route::post('/admin-charts', [App\Http\Controllers\AdminChartController::class, 'store'])->name('admin.charts.store');
});

// Public/User Chart Data
Route::get('/api/chart-data', [App\Http\Controllers\AdminChartController::class, 'getChartData'])->name('api.chart');

// ------------------------
// User Routes
// ------------------------
Route::middleware(['auth', 'verified', 'user'])->group(function () {
    // User Dashboard (with wallet & trades)
    Route::get('/user-dashboard', [TradeController::class, 'dashboard'])->name('user.dashboard');

    // Wallet Page
    Route::get('/wallet', [WalletTransactionController::class, 'userWallet'])->name('wallet');

    // Submit Deposit / Withdraw
    Route::post('/wallet/transaction', [WalletTransactionController::class, 'store'])->name('wallet.transaction');
    
    // Get Wallet Balance (Polling)
    Route::get('/wallet/balance', [WalletTransactionController::class, 'getBalance'])->name('wallet.balance');

    // Buy/Sell Trade
    Route::post('/trade', [TradeController::class, 'placeTrade'])->name('trade.place');

    // Fetch Latest Trades (optional, for real-time dashboard updates)
    Route::get('/latest-trades', [TradeController::class, 'latestTrades'])->name('trade.latest');
    
    // Customer Support Chat (User)
    Route::get('/chat/sync', [App\Http\Controllers\ChatController::class, 'getUserChat'])->name('chat.sync');
    Route::post('/chat/send', [App\Http\Controllers\ChatController::class, 'sendMessage'])->name('chat.send');

    // KYC Submission
    Route::post('/kyc/submit', [App\Http\Controllers\KycController::class, 'store'])->name('kyc.store');
    

});

// TEMPORARY DEBUG ROUTE - Remove after testing
Route::post('/test-deposit-direct', function(Illuminate\Http\Request $request) {
    \Log::info('=== TEST DEPOSIT ROUTE HIT ===', [
        'all_data' => $request->all(),
        'user' => auth()->user() ? auth()->user()->toArray() : 'not logged in',
    ]);
    
    try {
        $transaction = \App\Models\WalletTransaction::create([
            'user_id' => auth()->id() ?? 1,
            'crypto' => 'BTC',
            'amount' => 999.99,
            'type' => 'deposit',
            'status' => 'pending',
            'proof' => null,
        ]);
        
        \Log::info('Test transaction created', ['id' => $transaction->id]);
        
        return response()->json([
            'success' => true,
            'message' => 'Test deposit created',
            'transaction_id' => $transaction->id,
        ]);
    } catch (\Exception $e) {
        \Log::error('Test deposit failed', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);
        
        return response()->json([
            'success' => false,
            'error' => $e->getMessage(),
        ], 500);
    }
})->middleware(['web']);
Route::get('/about-us', function () {
    return Inertia::render('AboutUs'); // This matches AboutUs.jsx
});
Route::get('/contact-us', function () {
    return inertia('ContactUs');
})->name('contact');
// ------------------------
// Profile Routes
// ------------------------
Route::middleware(['auth', 'verified'])->group(function () {
    Route::get('/profile', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::patch('/profile', [ProfileController::class, 'update'])->name('profile.update');
    Route::delete('/profile', [ProfileController::class, 'destroy'])->name('profile.destroy');
});

// ------------------------
// Authentication Routes
// ------------------------
// TEMPORARY DEBUG ROUTE
Route::get('/debug-db', function() {
    $pdo = DB::connection()->getPdo();
    $serverInfo = DB::select('SELECT @@hostname as host, @@port as port, @@datadir as datadir, @@version as version')[0];
    $count = \App\Models\WalletTransaction::count();
    $latest = \App\Models\WalletTransaction::latest()->take(5)->get();
    
    return response()->json([
        'laravel_connection' => [
            'database' => config('database.connections.mysql.database'),
            'host' => config('database.connections.mysql.host'),
            'port' => config('database.connections.mysql.port'),
        ],
        'mysql_server_info' => $serverInfo,
        'data_check' => [
            'total_transactions' => $count,
            'latest_transactions' => $latest
        ]
    ]);
});

require __DIR__.'/auth.php';
