
import { useState, useEffect, useMemo } from 'react';
import { QRCodeCanvas } from 'qrcode.react';
import { router } from '@inertiajs/react';
import Navbar from '../Components/Navbar';
import { AreaChart, Area, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer } from 'recharts';

export default function Wallet({ auth, wallet, trades, errors }) {
    const user = auth?.user || {};
    const cryptos = ['BTC', 'ETH', 'SOL', 'XRP', 'DOGE', 'USDT', 'USDC'];

    // Debug logging
    console.log('=== WALLET PAGE DEBUG ===');
    console.log('User:', user);
    console.log('Wallet:', wallet);
    console.log('Trades:', trades);
    console.log('Errors:', errors);
    console.log('========================');

    const [selectedCrypto, setSelectedCrypto] = useState('BTC');
    const [activeTab, setActiveTab] = useState('deposit');
    const [amount, setAmount] = useState('');
    const [proof, setProof] = useState(null);
    const [currentWalletAddress, setCurrentWalletAddress] = useState('');
    const [withdrawalAddress, setWithdrawalAddress] = useState(''); // New state for user input
    const [successMessage, setSuccessMessage] = useState('');
    const [showWalletSection, setShowWalletSection] = useState(false);
    const [validationError, setValidationError] = useState('');
    const [showWithdrawConfirmation, setShowWithdrawConfirmation] = useState(false);
    const [currentQuote, setCurrentQuote] = useState('');

    const walletAddresses = {
        BTC: '3JqZWqHEecwiQuGnZAgdSackgNL1qWkAQq',
        ETH: '0xa250afe5289aa07578903faa663fc915ee8ffffd',
        SOL: '4oB8SBymq5CdyhN1nAiscfsYL4T9QffGJUuknMgH5mHV',
        XRP: 'rBuZfn1m4tA6znziHsRp9AyC1M3qg6rgbF',

        DOGE: '9vNWE1PrfLrUQs7mKnDPfPHZ23Ev8mES1u',
        USDT: 'TNibKBzQAcwurnCDdv89E2WMo1WdtN8sWF',
        USDC: '0xa250afe5289aa07578903faa663fc915ee8ffffd',
    };

    useEffect(() => {
        setCurrentWalletAddress(walletAddresses[selectedCrypto]);
    }, [selectedCrypto]);

    // Polling for real-time balance and transaction history (User Side)
    useEffect(() => {
        const interval = setInterval(() => {
            router.reload({ only: ['wallet', 'trades'], preserveScroll: true });
        }, 5000); // Poll every 5s

        return () => clearInterval(interval);
    }, []);

    // Handle backend validation errors
    useEffect(() => {
        if (errors && errors.amount) {
            setValidationError(errors.amount);
            setTimeout(() => setValidationError(''), 5000);
        }
    }, [errors]);

    const exchangeRates = {
        BTC: 65000,
        ETH: 3500,
        SOL: 150,
        XRP: 0.6,
        DOGE: 0.15,
        USDT: 1,
        USDC: 1,
    };

    const processTransaction = () => {
        const payload = new FormData();
        payload.append('crypto', selectedCrypto);
        payload.append('amount', amount);
        payload.append('type', activeTab);
        if (activeTab === 'withdraw') {
            payload.append('wallet_address', withdrawalAddress);
        }
        if (proof) payload.append('proof', proof);

        console.log('=== SENDING TO BACKEND ===');
        console.log('Crypto:', selectedCrypto);
        console.log('Amount:', amount);
        console.log('Type:', activeTab);
        console.log('Proof:', proof ? proof.name : 'No proof');

        router.post('/wallet/transaction', payload, {
            onSuccess: () => {
                console.log('✅ SUCCESS: Transaction submitted');
                setSuccessMessage(
                    activeTab === 'deposit'
                        ? 'Deposit request submitted!'
                        : 'Withdrawal request submitted!'
                );
                setAmount('');
                setProof(null);
                setShowWalletSection(false);
                setShowWithdrawConfirmation(false);
                setWithdrawalAddress('');

                setTimeout(() => setSuccessMessage(''), 3000);
            },
            onError: (errors) => {
                console.log('❌ ERROR:', errors);
                setShowWithdrawConfirmation(false);
                if (errors.amount) {
                    setValidationError(errors.amount);
                    setTimeout(() => setValidationError(''), 5000);
                }
            }
        });
    };

    const handleSubmit = (e) => {
        e.preventDefault();

        console.log('=== FORM SUBMIT ===');
        console.log('Active Tab:', activeTab);
        console.log('Amount:', amount);
        console.log('Proof:', proof);
        console.log('Selected Crypto:', selectedCrypto);

        // Validation - only check if amount is entered
        if (!amount || amount <= 0) {
            setValidationError('Please enter a valid amount');
            setTimeout(() => setValidationError(''), 3000);
            return;
        }

        // For withdrawals, check KYC then show confirmation
        if (activeTab === 'withdraw') {
            if (user.kyc_status !== 'approved') {
                let msg = `First register your account KYC (Status: ${user.kyc_status || 'unknown'})`;
                if (user.kyc_status === 'pending') {
                    msg = "Your KYC verification is pending approval";
                } else if (user.kyc_status === 'rejected') {
                    msg = "Your KYC verification was rejected. Please check your email/profile.";
                }

                setValidationError(msg);
                setTimeout(() => setValidationError(''), 4000);
                return;
            }

            if (!withdrawalAddress) {
                setValidationError('Please enter your wallet address for withdrawal');
                setTimeout(() => setValidationError(''), 3000);
                return;
            }

            // Set a random motivational quote
            const quotes = [
                "The market rewards patience. Are you sure you want to pull out now?",
                "Opportunities don't happen, you create them. Keep your capital ready!",
                "The biggest risk is not taking any risk. Keep trading!",
                "Compound interest is the eighth wonder of the world. Let your money grow!",
                "Winners never quit, and quitters never win. Stay in the game!",
                "Volatility is where the profit is. Don't miss the next big move!",
                "Diamond hands build wealth. Paper hands pay fees.",
                "Every withdrawal is a missed opportunity for the next moonshot!"
            ];
            setCurrentQuote(quotes[Math.floor(Math.random() * quotes.length)]);
            setShowWithdrawConfirmation(true);
            return;
        }

        console.log('Validation passed, submitting...');
        processTransaction();
    };

    const copyAddress = () => {
        navigator.clipboard.writeText(currentWalletAddress);
        alert('Address Copied!');
    };

    // Total Balance Calculation
    const totalBalanceUSDT = cryptos.reduce((acc, coin) => {
        const amount = Number(wallet[coin.toUpperCase()] || 0);
        return acc + amount;
    }, 0);

    // Real-time Chart Data Processing
    const chartData = useMemo(() => {
        if (!trades || trades.length === 0) return [];

        // 1. Sort trades by date (ascending)
        const sortedTrades = [...trades].sort((a, b) => new Date(a.created_at) - new Date(b.created_at));

        // 2. Aggregate cumulative value over time
        let cumulativeValue = 0;
        const dataPoints = [];
        const processedDates = new Set();

        sortedTrades.forEach(trade => {
            const date = new Date(trade.created_at).toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
            const value = parseFloat(trade.amount) * (exchangeRates[trade.crypto] || 0);

            if (trade.type === 'deposit' && trade.status === 'approved') {
                cumulativeValue += value;
            } else if (trade.type === 'withdraw' && trade.status === 'approved') {
                cumulativeValue -= value;
            }

            // Only add one data point per day (taking the last value of the day)
            // Ideally we'd group by day, but for simplicity we'll just push and let Recharts handle it or filter duplicates
            // A better approach for a smooth chart:
            if (!processedDates.has(date)) {
                dataPoints.push({ name: date, value: cumulativeValue });
                processedDates.add(date);
            } else {
                // Update the last entry for this date
                const lastPoint = dataPoints[dataPoints.length - 1];
                if (lastPoint.name === date) {
                    lastPoint.value = cumulativeValue;
                }
            }
        });

        // Ensure we have at least a few points for a nice line, or pad with initial 0
        if (dataPoints.length === 0 && totalBalanceUSDT > 0) {
            dataPoints.push({ name: 'Start', value: 0 });
            dataPoints.push({ name: 'Now', value: totalBalanceUSDT });
        }

        return dataPoints;
    }, [trades, totalBalanceUSDT]);

    return (
        <div className="min-h-screen bg-gray-50 dark:bg-gray-900 text-gray-900 dark:text-gray-100 flex flex-col">
            <Navbar user={user} />

            {/* Validation Popup */}
            {validationError && (
                <div className="fixed inset-0 flex items-center justify-center z-50">
                    <div className="px-8 py-4 rounded-xl shadow-lg text-white text-lg font-semibold bg-yellow-500 animate-slide-in">
                        {validationError}
                    </div>
                </div>
            )}

            {/* Withdrawal Confirmation Popup */}
            {showWithdrawConfirmation && (
                <div className="fixed inset-0 flex items-center justify-center z-50 bg-black/50 backdrop-blur-sm animate-fade-in">
                    <div className="bg-white dark:bg-gray-800 p-8 rounded-3xl shadow-2xl max-w-md w-full text-center transform transition-all scale-100">
                        <div className="w-20 h-20 bg-yellow-100 dark:bg-yellow-900/30 rounded-full flex items-center justify-center mx-auto mb-6">
                            <span className="text-4xl">🚀</span>
                        </div>
                        <h3 className="text-2xl font-bold mb-2">Wait! Are you sure?</h3>
                        <p className="text-gray-500 dark:text-gray-400 mb-8 italic text-lg px-4">
                            "{currentQuote || "Trading more could lead to higher profits. Why withdraw now when the market is moving?"}"
                        </p>
                        <div className="flex flex-col gap-3">
                            <button
                                onClick={() => setShowWithdrawConfirmation(false)}
                                className="w-full py-3 rounded-xl font-bold text-white bg-gradient-to-r from-indigo-500 to-purple-600 hover:shadow-lg hover:scale-105 transition-all duration-300"
                            >
                                Keep Trading & Earn More!
                            </button>
                            <button
                                onClick={processTransaction}
                                className="w-full py-3 rounded-xl font-semibold text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 transition-colors"
                            >
                                Proceed to Withdraw
                            </button>
                        </div>
                    </div>
                </div>
            )}

            {/* Success Popup */}
            {successMessage && (
                <div className="fixed inset-0 flex items-center justify-center z-50">
                    <div className={`px-8 py-4 rounded-xl shadow-lg text-white text-lg font-semibold ${activeTab === 'deposit' ? 'bg-green-500' : 'bg-red-500'} animate-slide-in`}>
                        {successMessage}
                    </div>
                </div>
            )}

            <div className="flex-1 w-full px-6 lg:px-12 py-8 space-y-8">

                {/* Top Section: Total Balance & Chart */}
                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    {/* Total Balance Card */}
                    <div className="lg:col-span-1 bg-gradient-to-br from-blue-600 to-purple-600 rounded-3xl shadow-2xl p-8 text-white flex flex-col justify-center relative overflow-hidden">
                        <div className="relative z-10">
                            <h2 className="text-xl font-medium opacity-80 mb-2">Total Balance</h2>
                            <h1 className="text-5xl font-bold mb-4">${totalBalanceUSDT.toFixed(2)} <span className="text-2xl opacity-60">USDT</span></h1>
                            <p className="opacity-70 text-sm">Estimated value of all assets</p>
                        </div>
                        {/* Decorative Circle */}
                        <div className="absolute -bottom-10 -right-10 w-40 h-40 bg-white opacity-10 rounded-full blur-2xl"></div>
                    </div>

                    {/* Portfolio Chart */}
                    <div className="lg:col-span-2 bg-white dark:bg-gray-800 rounded-3xl shadow-xl p-6">
                        <h3 className="text-lg font-bold mb-4">Portfolio Growth</h3>
                        <div className="h-64 w-full">
                            {chartData.length > 0 ? (
                                <ResponsiveContainer width="100%" height="100%">
                                    <AreaChart data={chartData}>
                                        <defs>
                                            <linearGradient id="colorValue" x1="0" y1="0" x2="0" y2="1">
                                                <stop offset="5%" stopColor="#8884d8" stopOpacity={0.8} />
                                                <stop offset="95%" stopColor="#8884d8" stopOpacity={0} />
                                            </linearGradient>
                                        </defs>
                                        <CartesianGrid strokeDasharray="3 3" vertical={false} opacity={0.1} />
                                        <XAxis dataKey="name" axisLine={false} tickLine={false} />
                                        <YAxis axisLine={false} tickLine={false} />
                                        <Tooltip contentStyle={{ borderRadius: '10px', border: 'none', boxShadow: '0 4px 12px rgba(0,0,0,0.1)' }} />
                                        <Area type="monotone" dataKey="value" stroke="#8884d8" fillOpacity={1} fill="url(#colorValue)" />
                                    </AreaChart>
                                </ResponsiveContainer>
                            ) : (
                                <div className="h-full flex flex-col items-center justify-center text-gray-400 opacity-60">
                                    <svg className="w-16 h-16 mb-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="1.5" d="M7 12l3-3 3 3 4-4M8 21l4-4 4 4M3 4h18M4 4h16v12a1 1 0 01-1 1H5a1 1 0 01-1-1V4z"></path></svg>
                                    <p>No trading activity yet</p>
                                </div>
                            )}
                        </div>
                    </div>
                </div>

                {/* Main Content: Crypto List & Wallet Action */}
                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8 items-start">

                    {/* Left: Crypto List (CTAs) */}
                    <div className="lg:col-span-1 space-y-4">
                        <h3 className="text-xl font-bold mb-4">Assets</h3>
                        <div className="grid grid-cols-2 lg:grid-cols-1 gap-4">
                            {cryptos.map((crypto) => (
                                <button
                                    key={crypto}
                                    onClick={() => {
                                        setSelectedCrypto(crypto);
                                        setShowWalletSection(true);
                                    }}
                                    className={`w-full text-left p-4 rounded-xl shadow-md transition-all duration-300 transform hover:scale-102 hover:shadow-lg flex items-center justify-between group ${selectedCrypto === crypto && showWalletSection
                                        ? 'bg-gradient-to-r from-indigo-500 to-purple-500 text-white ring-2 ring-offset-2 ring-indigo-500'
                                        : 'bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700'
                                        }`}
                                >
                                    <div>
                                        <span className="font-bold text-lg">{crypto}</span>
                                        <span className="block text-xs opacity-60 group-hover:opacity-100 transition-opacity">Click to manage</span>
                                    </div>
                                    <div className="text-right">
                                        {/* Arrow Icon */}
                                        <svg xmlns="http://www.w3.org/2000/svg" className={`h-6 w-6 transition-transform ${selectedCrypto === crypto && showWalletSection ? 'rotate-90' : ''}`} fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                                        </svg>
                                    </div>
                                </button>
                            ))}
                        </div>
                    </div>

                    {/* Right: Wallet Action Section (Conditional) */}
                    <div className="lg:col-span-2">
                        {showWalletSection ? (
                            <div className="bg-white dark:bg-gray-800 rounded-3xl shadow-2xl p-8 animate-fade-in relative overflow-hidden">
                                <div className="flex flex-col md:flex-row gap-8">
                                    {/* QR Code Section */}
                                    <div className="flex flex-col items-center justify-center space-y-4 p-4 bg-gray-50 dark:bg-gray-700/30 rounded-2xl">
                                        <div className="bg-white p-2 rounded-xl shadow-sm">
                                            <QRCodeCanvas value={currentWalletAddress} size={180} />
                                        </div>
                                        <button
                                            onClick={copyAddress}
                                            className="text-sm font-semibold text-indigo-600 dark:text-indigo-400 hover:underline"
                                        >
                                            Copy Address
                                        </button>
                                    </div>

                                    {/* Action Form */}
                                    <div className="flex-1 space-y-6">
                                        <div>
                                            <h2 className="text-2xl font-bold flex items-center gap-2">
                                                {selectedCrypto} Wallet
                                                <span className="text-sm font-normal bg-gray-100 dark:bg-gray-700 px-2 py-1 rounded-lg">
                                                    Bal: {wallet[selectedCrypto.toUpperCase()] || 0}
                                                </span>
                                            </h2>
                                            <p className="text-xs text-gray-500 mt-1 break-all font-mono bg-gray-100 dark:bg-gray-900 p-2 rounded select-all">
                                                {currentWalletAddress}
                                            </p>
                                        </div>

                                        {/* Tabs */}
                                        <div className="flex p-1 bg-gray-100 dark:bg-gray-700 rounded-xl">
                                            {['deposit', 'withdraw'].map((tab) => (
                                                <button
                                                    key={tab}
                                                    onClick={() => setActiveTab(tab)}
                                                    className={`flex-1 py-2 rounded-lg text-sm font-bold transition-all ${activeTab === tab
                                                        ? tab === 'deposit'
                                                            ? 'bg-white dark:bg-gray-600 text-green-600 shadow-sm'
                                                            : 'bg-white dark:bg-gray-600 text-red-600 shadow-sm'
                                                        : 'text-gray-500 hover:text-gray-700 dark:hover:text-gray-300'
                                                        }`}
                                                >
                                                    {tab.toUpperCase()}
                                                </button>
                                            ))}
                                        </div>

                                        <form onSubmit={handleSubmit} className="space-y-4">
                                            <div>
                                                <label className="block text-xs font-semibold text-gray-500 mb-1">Amount</label>
                                                <input
                                                    type="number"
                                                    step="any"
                                                    min="0"
                                                    value={amount}
                                                    onChange={(e) => setAmount(e.target.value)}
                                                    placeholder={`Enter ${selectedCrypto} amount`}
                                                    className="w-full bg-gray-50 dark:bg-gray-900 border border-gray-200 dark:border-gray-700 rounded-xl px-4 py-3 focus:ring-2 focus:ring-indigo-500 outline-none transition-all"
                                                />
                                            </div>

                                            {activeTab === 'withdraw' && (
                                                <div>
                                                    <label className="block text-xs font-semibold text-gray-500 mb-1">Your Wallet Address</label>
                                                    <input
                                                        type="text"
                                                        value={withdrawalAddress}
                                                        onChange={(e) => setWithdrawalAddress(e.target.value)}
                                                        placeholder={`Paste your ${selectedCrypto} wallet address`}
                                                        className="w-full bg-gray-50 dark:bg-gray-900 border border-gray-200 dark:border-gray-700 rounded-xl px-4 py-3 focus:ring-2 focus:ring-indigo-500 outline-none transition-all"
                                                        required
                                                    />
                                                    <div className="mt-2 flex items-start gap-2 p-3 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg">
                                                        <svg className="w-5 h-5 text-yellow-600 dark:text-yellow-500 flex-shrink-0 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                                                            <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                                        </svg>
                                                        <div className="text-xs text-yellow-800 dark:text-yellow-200">
                                                            <p className="font-semibold mb-1">⚠️ Important Reminder:</p>
                                                            <p>Please double-check your {selectedCrypto} wallet address before submitting. Sending funds to an incorrect address may result in permanent loss. We cannot reverse or refund transactions sent to wrong addresses.</p>
                                                        </div>
                                                    </div>
                                                </div>
                                            )}

                                            {activeTab === 'deposit' && (
                                                <div>
                                                    <label className="block text-xs font-semibold text-gray-500 mb-1">Payment Proof</label>
                                                    <label className="flex flex-col items-center justify-center w-full h-32 border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-xl cursor-pointer hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors">
                                                        <div className="flex flex-col items-center justify-center pt-5 pb-6">
                                                            <svg className="w-8 h-8 mb-3 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"></path></svg>
                                                            <p className="text-sm text-gray-500">{proof ? proof.name : 'Click to upload proof'}</p>
                                                        </div>
                                                        <input type="file" className="hidden" onChange={(e) => setProof(e.target.files[0])} />
                                                    </label>
                                                </div>
                                            )}

                                            <button
                                                type="submit"
                                                className={`w-full py-4 rounded-xl font-bold text-white shadow-lg transform transition hover:scale-105 active:scale-95 ${activeTab === 'deposit'
                                                    ? 'bg-gradient-to-r from-green-500 to-emerald-600 shadow-green-500/30'
                                                    : 'bg-gradient-to-r from-red-500 to-rose-600 shadow-red-500/30'
                                                    }`}
                                            >
                                                {activeTab === 'deposit' ? 'Confirm Deposit' : 'Confirm Withdraw'}
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        ) : (
                            <div className="h-full min-h-[400px] flex flex-col items-center justify-center bg-white dark:bg-gray-800 rounded-3xl shadow-xl p-8 text-center opacity-60 border-2 border-dashed border-gray-200 dark:border-gray-700">
                                <div className="w-20 h-20 bg-gray-100 dark:bg-gray-700 rounded-full flex items-center justify-center mb-4">
                                    <svg className="w-10 h-10 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path></svg>
                                </div>
                                <h3 className="text-xl font-bold text-gray-600 dark:text-gray-300">Select a Crypto Asset</h3>
                                <p className="text-gray-500 max-w-xs mt-2">Choose a cryptocurrency from the list to view your wallet address and manage transactions.</p>
                            </div>
                        )}
                    </div>
                </div>

                {/* Transaction History */}
                <div className="bg-white dark:bg-gray-800 rounded-3xl shadow-xl p-8">
                    <h2 className="text-2xl font-bold mb-6">Transaction History</h2>
                    <div className="overflow-x-auto">
                        <table className="min-w-full">
                            <thead>
                                <tr className="border-b border-gray-100 dark:border-gray-700">
                                    <th className="px-6 py-4 text-left text-xs font-semibold text-gray-500 uppercase tracking-wider">Date</th>
                                    <th className="px-6 py-4 text-left text-xs font-semibold text-gray-500 uppercase tracking-wider">Crypto</th>
                                    <th className="px-6 py-4 text-left text-xs font-semibold text-gray-500 uppercase tracking-wider">Amount</th>
                                    <th className="px-6 py-4 text-left text-xs font-semibold text-gray-500 uppercase tracking-wider">Type</th>
                                    <th className="px-6 py-4 text-left text-xs font-semibold text-gray-500 uppercase tracking-wider">Status</th>
                                </tr>
                            </thead>
                            <tbody className="divide-y divide-gray-100 dark:divide-gray-700">
                                {trades.map((trade) => (
                                    <tr key={trade.id} className="hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors">
                                        <td className="px-6 py-4 text-sm text-gray-900 dark:text-gray-100">{new Date(trade.created_at).toLocaleDateString()}</td>
                                        <td className="px-6 py-4 text-sm font-medium">{trade.crypto}/USDT</td>
                                        <td className="px-6 py-4 text-sm">{trade.amount}</td>
                                        <td className="px-6 py-4">
                                            <span className={`px-3 py-1 rounded-full text-xs font-medium ${trade.type === 'deposit' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                                                }`}>
                                                {trade.type}
                                            </span>
                                        </td>
                                        <td className="px-6 py-4">
                                            <span className={`px-3 py-1 rounded-full text-xs font-medium ${trade.status === 'approved' ? 'bg-blue-100 text-blue-800' : 'bg-yellow-100 text-yellow-800'
                                                }`}>
                                                {trade.status}
                                            </span>
                                        </td>
                                    </tr>
                                ))}
                                {trades.length === 0 && (
                                    <tr>
                                        <td colSpan="5" className="px-6 py-8 text-center text-gray-500">No transactions found</td>
                                    </tr>
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>

            </div>
        </div>
    );
}
