import { useState } from 'react';
import { router, useForm, Link } from '@inertiajs/react';
import KycForm from '@/Components/KycForm';

export default function Profile({ auth, mustVerifyEmail, status, kyc }) {
    const user = auth?.user || {};
    const [activeTab, setActiveTab] = useState('info');

    // Profile Information Form
    const { data: profileData, setData: setProfileData, patch: patchProfile, processing: processingProfile, errors: profileErrors } = useForm({
        name: user.name || '',
        email: user.email || '',
    });

    // Password Form
    const { data: passwordData, setData: setPasswordData, put: putPassword, processing: processingPassword, errors: passwordErrors, reset: resetPassword } = useForm({
        current_password: '',
        password: '',
        password_confirmation: '',
    });

    const handleProfileSubmit = (e) => {
        e.preventDefault();
        patchProfile(route('profile.update'), {
            preserveScroll: true,
            onSuccess: () => {
                alert('Profile updated successfully!');
            },
        });
    };

    const handlePasswordSubmit = (e) => {
        e.preventDefault();
        putPassword(route('password.update'), {
            preserveScroll: true,
            onSuccess: () => {
                resetPassword();
                alert('Password updated successfully!');
            },
        });
    };

    const handleDeleteAccount = () => {
        if (confirm('Are you sure you want to delete your account? This action cannot be undone.')) {
            router.delete(route('profile.destroy'));
        }
    };

    return (
        <div className="min-h-screen bg-gray-50 dark:bg-gray-900 text-gray-900 dark:text-gray-100">
            {/* Back Button */}
            <div className="bg-gradient-to-r from-purple-600 via-blue-600 to-indigo-600 text-white shadow-xl sticky top-0 z-50">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4">
                    <button
                        onClick={() => window.history.back()}
                        className="flex items-center space-x-2 text-white hover:text-yellow-400 transition-colors duration-300"
                    >
                        <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 19l-7-7m0 0l7-7m-7 7h18" />
                        </svg>
                        <span className="font-semibold text-lg">Back</span>
                    </button>
                </div>
            </div>

            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
                {/* Header */}
                <div className="mb-8">
                    <h1 className="text-4xl font-bold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
                        Profile Settings
                    </h1>
                    <p className="text-gray-600 dark:text-gray-400 mt-2">
                        Manage your account settings and preferences
                    </p>
                </div>

                {/* Tabs */}
                <div className="flex space-x-1 bg-gray-200 dark:bg-gray-800 p-1 rounded-xl mb-8 overflow-x-auto">
                    {['info', 'password', 'kyc', 'danger'].map((tab) => (
                        <button
                            key={tab}
                            onClick={() => setActiveTab(tab)}
                            className={`flex-1 py-3 px-6 rounded-lg font-semibold transition-all duration-300 whitespace-nowrap ${activeTab === tab
                                ? 'bg-white dark:bg-gray-700 text-blue-600 dark:text-blue-400 shadow-md'
                                : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
                                }`}
                        >
                            {tab === 'info' && 'Profile Information'}
                            {tab === 'password' && 'Change Password'}
                            {tab === 'kyc' && 'Identity Verification'}
                            {tab === 'danger' && 'Danger Zone'}
                        </button>
                    ))}
                </div>

                {/* Profile Information Tab */}
                {activeTab === 'info' && (
                    <div className="bg-white dark:bg-gray-800 rounded-3xl shadow-xl p-8 animate-fade-in">
                        <h2 className="text-2xl font-bold mb-6">Profile Information</h2>
                        <p className="text-gray-600 dark:text-gray-400 mb-6">
                            Update your account profile information and email address.
                        </p>

                        <form onSubmit={handleProfileSubmit} className="space-y-6">
                            {/* Name */}
                            <div>
                                <label className="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    Name
                                </label>
                                <input
                                    type="text"
                                    value={profileData.name}
                                    onChange={(e) => setProfileData('name', e.target.value)}
                                    className="w-full px-4 py-3 rounded-xl bg-gray-50 dark:bg-gray-900 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all"
                                    required
                                />
                                {profileErrors.name && (
                                    <p className="text-red-500 text-sm mt-1">{profileErrors.name}</p>
                                )}
                            </div>

                            {/* Email */}
                            <div>
                                <label className="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    Email
                                </label>
                                <input
                                    type="email"
                                    value={profileData.email}
                                    onChange={(e) => setProfileData('email', e.target.value)}
                                    className="w-full px-4 py-3 rounded-xl bg-gray-50 dark:bg-gray-900 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all"
                                    required
                                />
                                {profileErrors.email && (
                                    <p className="text-red-500 text-sm mt-1">{profileErrors.email}</p>
                                )}
                            </div>

                            {/* Email Verification Notice */}
                            {mustVerifyEmail && user.email_verified_at === null && (
                                <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-xl p-4">
                                    <p className="text-yellow-800 dark:text-yellow-200 text-sm">
                                        Your email address is unverified. Please check your inbox for a verification link.
                                    </p>
                                </div>
                            )}

                            {/* Submit Button */}
                            <button
                                type="submit"
                                disabled={processingProfile}
                                className="w-full py-3 px-6 bg-gradient-to-r from-blue-500 to-purple-600 text-white font-bold rounded-xl shadow-lg hover:shadow-xl transform hover:scale-105 active:scale-95 transition-all duration-300 disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                {processingProfile ? 'Saving...' : 'Save Changes'}
                            </button>
                        </form>
                    </div>
                )}

                {/* Change Password Tab */}
                {activeTab === 'password' && (
                    <div className="bg-white dark:bg-gray-800 rounded-3xl shadow-xl p-8 animate-fade-in">
                        <h2 className="text-2xl font-bold mb-6">Change Password</h2>
                        <p className="text-gray-600 dark:text-gray-400 mb-6">
                            Ensure your account is using a long, random password to stay secure.
                        </p>

                        <form onSubmit={handlePasswordSubmit} className="space-y-6">
                            {/* Current Password */}
                            <div>
                                <label className="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    Current Password
                                </label>
                                <input
                                    type="password"
                                    value={passwordData.current_password}
                                    onChange={(e) => setPasswordData('current_password', e.target.value)}
                                    className="w-full px-4 py-3 rounded-xl bg-gray-50 dark:bg-gray-900 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all"
                                    required
                                />
                                {passwordErrors.current_password && (
                                    <p className="text-red-500 text-sm mt-1">{passwordErrors.current_password}</p>
                                )}
                            </div>

                            {/* New Password */}
                            <div>
                                <label className="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    New Password
                                </label>
                                <input
                                    type="password"
                                    value={passwordData.password}
                                    onChange={(e) => setPasswordData('password', e.target.value)}
                                    className="w-full px-4 py-3 rounded-xl bg-gray-50 dark:bg-gray-900 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all"
                                    required
                                />
                                {passwordErrors.password && (
                                    <p className="text-red-500 text-sm mt-1">{passwordErrors.password}</p>
                                )}
                            </div>

                            {/* Confirm Password */}
                            <div>
                                <label className="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">
                                    Confirm Password
                                </label>
                                <input
                                    type="password"
                                    value={passwordData.password_confirmation}
                                    onChange={(e) => setPasswordData('password_confirmation', e.target.value)}
                                    className="w-full px-4 py-3 rounded-xl bg-gray-50 dark:bg-gray-900 border border-gray-200 dark:border-gray-700 focus:ring-2 focus:ring-blue-500 outline-none transition-all"
                                    required
                                />
                            </div>

                            {/* Submit Button */}
                            <button
                                type="submit"
                                disabled={processingPassword}
                                className="w-full py-3 px-6 bg-gradient-to-r from-green-500 to-emerald-600 text-white font-bold rounded-xl shadow-lg hover:shadow-xl transform hover:scale-105 active:scale-95 transition-all duration-300 disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                {processingPassword ? 'Updating...' : 'Update Password'}
                            </button>
                        </form>
                    </div>
                )}

                {/* KYC Verification Tab */}
                {activeTab === 'kyc' && (
                    <div className="animate-fade-in">
                        <KycForm
                            status={user.kyc_status || 'unverified'}
                            rejectionReason={kyc?.rejection_reason}
                        />
                    </div>
                )}

                {/* Danger Zone Tab */}
                {activeTab === 'danger' && (
                    <div className="bg-white dark:bg-gray-800 rounded-3xl shadow-xl p-8 animate-fade-in border-2 border-red-200 dark:border-red-900">
                        <h2 className="text-2xl font-bold mb-6 text-red-600 dark:text-red-400">Danger Zone</h2>
                        <p className="text-gray-600 dark:text-gray-400 mb-6">
                            Once your account is deleted, all of its resources and data will be permanently deleted. Before deleting your account, please download any data or information that you wish to retain.
                        </p>

                        <button
                            onClick={handleDeleteAccount}
                            className="w-full py-3 px-6 bg-gradient-to-r from-red-500 to-rose-600 text-white font-bold rounded-xl shadow-lg hover:shadow-xl transform hover:scale-105 active:scale-95 transition-all duration-300"
                        >
                            Delete Account
                        </button>
                    </div>
                )}
            </div>

            <style>{`
                @keyframes fade-in {
                    from { opacity: 0; transform: translateY(10px); }
                    to { opacity: 1; transform: translateY(0); }
                }
                .animate-fade-in {
                    animation: fade-in 0.3s ease-out;
                }
            `}</style>
        </div>
    );
}
