import React, { useEffect, useState } from "react";
import Navbar from "../Components/Navbar";
import Hero from "../Components/Home/Hero";
import MarketTicker from "../Components/Home/MarketTicker";
import Features from "../Components/Home/Features";
import Testimonials from "../Components/Home/Testimonials";
import HomeFooter from "../Components/Home/HomeFooter";
import axios from "axios";

export default function Home({ auth }) {
  const user = auth?.user;
  const [topCryptos, setTopCryptos] = useState([]);
  const [selectedTab, setSelectedTab] = useState("gainers");

  useEffect(() => {
    const fetchData = async () => {
      try {
        const res = await axios.get("/market-data", {
          params: { per_page: 50 },
        });
        if (Array.isArray(res.data)) {
          setTopCryptos(res.data);
        } else {
          throw new Error("Invalid data format");
        }
      } catch (e) {
        console.error("CoinGecko API failed, using fallback data", e);
        // Fallback mock data to keep UI functional and beautiful
        setTopCryptos([
          { id: 'bitcoin', symbol: 'btc', name: 'Bitcoin', current_price: 65432.10, price_change_percentage_24h: 2.5, image: 'https://assets.coingecko.com/coins/images/1/large/bitcoin.png' },
          { id: 'ethereum', symbol: 'eth', name: 'Ethereum', current_price: 3456.78, price_change_percentage_24h: 1.8, image: 'https://assets.coingecko.com/coins/images/279/large/ethereum.png' },
          { id: 'solana', symbol: 'sol', name: 'Solana', current_price: 145.67, price_change_percentage_24h: -0.5, image: 'https://assets.coingecko.com/coins/images/4128/large/solana.png' },
          { id: 'binancecoin', symbol: 'bnb', name: 'BNB', current_price: 590.12, price_change_percentage_24h: 0.2, image: 'https://assets.coingecko.com/coins/images/825/large/binance-coin-logo.png' },
          { id: 'ripple', symbol: 'xrp', name: 'XRP', current_price: 0.61, price_change_percentage_24h: -1.2, image: 'https://assets.coingecko.com/coins/images/44/large/xrp-symbol-white-128.png' },
          { id: 'dogecoin', symbol: 'doge', name: 'Dogecoin', current_price: 0.16, price_change_percentage_24h: 5.4, image: 'https://assets.coingecko.com/coins/images/5/large/dogecoin.png' },
        ]);
      }
    };
    fetchData();
  }, []);

  const topGainers = [...topCryptos].sort((a, b) => b.price_change_percentage_24h - a.price_change_percentage_24h);
  const topLosers = [...topCryptos].sort((a, b) => a.price_change_percentage_24h - b.price_change_percentage_24h);

  return (
    <div className="min-h-screen flex flex-col bg-gray-50 dark:bg-black transition-colors duration-500 overflow-x-hidden font-sans">
      <Navbar user={user} />

      {/* Live Market Ticker */}
      <MarketTicker coins={topCryptos} />

      <main className="flex-1 w-full">
        {/* Hero Section */}
        <Hero user={user} topCoins={topCryptos.slice(0, 3)} />

        {/* Market Insights (Gainers/Losers) */}
        <section className="py-20 px-6 md:px-12 bg-white dark:bg-gray-900 border-y border-gray-100 dark:border-gray-800">
          <div className="container mx-auto">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">Market Movers</h2>
              <div className="flex justify-center gap-2 bg-gray-100 dark:bg-gray-800 p-1 rounded-xl inline-flex">
                <button
                  onClick={() => setSelectedTab('gainers')}
                  className={`px-6 py-2 rounded-lg text-sm font-bold transition-all ${selectedTab === 'gainers' ? 'bg-white dark:bg-gray-700 text-green-600 shadow-sm' : 'text-gray-500 hover:text-gray-900 dark:hover:text-white'}`}
                >
                  Top Gainers 🔥
                </button>
                <button
                  onClick={() => setSelectedTab('losers')}
                  className={`px-6 py-2 rounded-lg text-sm font-bold transition-all ${selectedTab === 'losers' ? 'bg-white dark:bg-gray-700 text-red-500 shadow-sm' : 'text-gray-500 hover:text-gray-900 dark:hover:text-white'}`}
                >
                  Top Losers 📉
                </button>
              </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {(selectedTab === "gainers" ? topGainers : topLosers).slice(0, 12).map((coin) => (
                <div key={coin.id} className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-800 rounded-xl hover:bg-gray-100 dark:hover:bg-gray-750 transition-colors cursor-pointer group">
                  <div className="flex items-center gap-4">
                    <span className="text-gray-400 font-mono text-sm w-4">{topCryptos.indexOf(coin) + 1}</span>
                    <img src={coin.image} alt={coin.name} className="w-10 h-10 rounded-full" />
                    <div>
                      <h4 className="font-bold text-gray-900 dark:text-white group-hover:text-blue-500 transition-colors">{coin.name}</h4>
                      <span className="text-xs text-gray-500 uppercase">{coin.symbol}</span>
                    </div>
                  </div>
                  <div className="text-right">
                    <div className="font-mono text-gray-900 dark:text-white font-medium">${coin.current_price.toLocaleString()}</div>
                    <div className={`text-sm font-bold ${coin.price_change_percentage_24h >= 0 ? 'text-green-500' : 'text-red-500'}`}>
                      {coin.price_change_percentage_24h > 0 ? '+' : ''}{coin.price_change_percentage_24h?.toFixed(2)}%
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </section>

        {/* Features Section */}
        <Features />

        {/* Call to Action Banner */}
        <section className="py-20 px-6">
          <div className="container mx-auto bg-gradient-to-r from-blue-600 to-indigo-700 rounded-3xl p-10 md:p-16 text-center text-white shadow-2xl relative overflow-hidden">
            <div className="absolute top-0 right-0 w-64 h-64 bg-white opacity-10 rounded-full blur-3xl translate-x-1/2 -translate-y-1/2"></div>
            <div className="absolute bottom-0 left-0 w-64 h-64 bg-white opacity-10 rounded-full blur-3xl -translate-x-1/2 translate-y-1/2"></div>

            <h2 className="text-3xl md:text-5xl font-bold mb-6 relative z-10">Start Your Crypto Journey Today</h2>
            <p className="text-blue-100 text-lg mb-8 max-w-2xl mx-auto relative z-10">
              Join thousands of traders worldwide. Sign up now to get access to advanced trading tools and lowest fees in the market.
            </p>
            <div className="relative z-10">
              {user ? (
                <a href="/wallet" className="inline-block px-10 py-4 bg-white text-blue-600 font-bold rounded-xl shadow-lg hover:bg-gray-100 transform hover:scale-105 transition-all">
                  Go to Dashboard
                </a>
              ) : (
                <a href="/register" className="inline-block px-10 py-4 bg-white text-blue-600 font-bold rounded-xl shadow-lg hover:bg-gray-100 transform hover:scale-105 transition-all">
                  Create Free Account
                </a>
              )}
            </div>
          </div>
        </section>

        {/* Testimonials */}
        <Testimonials />
      </main>

      <HomeFooter />
    </div>
  );
}
