import React, { useState, useEffect, useCallback, useRef } from "react";
import axios from "axios";
import { motion, AnimatePresence } from "framer-motion";
import Navbar from "../Components/Navbar";

// Simple TradingView Chart Component
function TradingChart({ symbol, theme }) {
  const chartContainerRef = useRef(null);
  const widgetRef = useRef(null);

  useEffect(() => {
    if (!chartContainerRef.current || typeof window === 'undefined') return;

    // Clear previous widget
    if (chartContainerRef.current) {
      chartContainerRef.current.innerHTML = '';
    }

    // Create TradingView widget script if not already loaded
    const loadTradingView = () => {
      if (window.TradingView && chartContainerRef.current) {
        widgetRef.current = new window.TradingView.widget({
          autosize: true,
          symbol: `BINANCE:${symbol}USDT`,
          interval: '60',
          timezone: 'Etc/UTC',
          theme: theme === 'dark' ? 'dark' : 'light',
          style: '1',
          locale: 'en',
          toolbar_bg: theme === 'dark' ? '#1f2937' : '#ffffff',
          enable_publishing: false,
          allow_symbol_change: true,
          container_id: chartContainerRef.current.id,
          height: 500,
          width: '100%',
        });
      }
    };

    if (window.TradingView) {
      loadTradingView();
    } else {
      const script = document.createElement('script');
      script.src = 'https://s3.tradingview.com/tv.js';
      script.async = true;
      script.onload = loadTradingView;
      document.head.appendChild(script);

      return () => {
        if (script.parentNode) {
          script.parentNode.removeChild(script);
        }
      };
    }

    return () => {
      if (chartContainerRef.current) {
        chartContainerRef.current.innerHTML = '';
      }
      widgetRef.current = null;
    };
  }, [symbol, theme]);

  return (
    <div
      id={`tradingview_${symbol}`}
      ref={chartContainerRef}
      className="w-full h-[400px] md:h-[500px] rounded-lg shadow-lg bg-white dark:bg-gray-800 overflow-hidden"
    />
  );
}

export default function Dashboard({ auth, wallet: initialWallet = {}, trades: initialTrades = [] }) {
  const user = auth?.user || {};
  const [wallet, setWallet] = useState(initialWallet);
  const [trades, setTrades] = useState(initialTrades || []);
  const [expandedTrade, setExpandedTrade] = useState(null);
  const [selectedCrypto, setSelectedCrypto] = useState("BTC");
  const [amount, setAmount] = useState("");
  const [leverage, setLeverage] = useState("");
  const [expiry, setExpiry] = useState("");
  const [loadingBuy, setLoadingBuy] = useState(false);
  const [loadingSell, setLoadingSell] = useState(false);
  const [error, setError] = useState("");
  const [showPopup, setShowPopup] = useState(false);
  const [popupMessage, setPopupMessage] = useState("");
  const [theme, setTheme] = useState(() => {
    if (typeof document !== 'undefined') {
      return document.documentElement.classList.contains("dark") ? "dark" : "light";
    }
    return "light";
  });
  const [marketPrices, setMarketPrices] = useState({});
  const isFetchingTrades = useRef(false);

  const marketList = ["BTC", "ETH", "SOL", "BNB", "XRP", "ADA", "DOGE", "TRX", "LINK", "DOT", "MATIC", "LTC", "AVAX", "ATOM", "ETC"];
  const expiryOptions = [
    { label: "1 min", value: 60 },
    { label: "2 min", value: 120 },
    { label: "12 hours", value: 43200 },
    { label: "1 day", value: 86400 },
    { label: "3 days", value: 259200 },
    { label: "7 days", value: 604800 },
  ];

  // Leverage to expiry mapping
  const leverageExpiryMap = {
    "1:10": 60,      // 1 min
    "1:35": 120,     // 2 min
    "1:87": 43200,   // 12 hours
    "1:165": 86400,  // 1 day
    "1:250": 259200, // 3 days
    "1:365": 604800, // 7 days
  };

  // Detect theme change
  useEffect(() => {
    if (typeof document === 'undefined') return;
    const observer = new MutationObserver(() => {
      if (typeof document !== 'undefined' && document.documentElement) {
        setTheme(document.documentElement.classList.contains("dark") ? "dark" : "light");
      }
    });
    observer.observe(document.documentElement, { attributes: true, attributeFilter: ["class"] });
    return () => observer.disconnect();
  }, []);

  // Fetch live market prices from CoinGecko API
  const fetchMarketPrices = async () => {
    const coinMap = {
      BTC: "bitcoin",
      ETH: "ethereum",
      SOL: "solana",
      BNB: "binancecoin",
      XRP: "ripple",
      ADA: "cardano",
      DOGE: "dogecoin",
      TRX: "tron",
      LINK: "chainlink",
      DOT: "polkadot",
      MATIC: "matic-network",
      LTC: "litecoin",
      AVAX: "avalanche-2",
      ATOM: "cosmos",
      ETC: "ethereum-classic",
    };

    try {
      const ids = Object.values(coinMap).join(",");
      // Use local proxy instead of direct CoinGecko call to avoid CORS
      const res = await axios.get("/market-data", {
        params: {
          ids: ids
        },
        timeout: 10000, // 10 second timeout
      });

      if (res.data && Array.isArray(res.data)) {
        console.log("Market Data Response:", res.data);
        const priceMap = {};
        res.data.forEach(item => {
          // Find the symbol key from our map based on the ID
          const symbol = Object.keys(coinMap).find(k => coinMap[k] === item.id);
          if (symbol) {
            priceMap[symbol] = {
              price: item.current_price?.toLocaleString() || "0",
              change: item.price_change_percentage_24h?.toFixed(2) || "0",
              image: item.image || null,
            };
          }
        });
        setMarketPrices(priceMap);
      } else {
        console.warn("Market data response is not an array:", res.data);
      }
    } catch (err) {
      console.error("Failed to fetch market prices", err);
      // Set empty prices to prevent crashes
      setMarketPrices({});
    }
  };

  useEffect(() => {
    fetchMarketPrices();
    const interval = setInterval(fetchMarketPrices, 30000);
    return () => clearInterval(interval);
  }, []);

  // Poll wallet balance for real-time updates
  const fetchWallet = async () => {
    try {
      const res = await axios.get("/wallet/balance");
      if (res.data.wallet) {
        setWallet(res.data.wallet);
      }
    } catch (err) {
      if (err.response?.status === 401) {
        window.location.href = "/login";
        return;
      }
      console.error("Failed to fetch wallet", err);
    }
  };

  useEffect(() => {
    fetchWallet();
    const interval = setInterval(fetchWallet, 3000); // Poll every 3s
    return () => clearInterval(interval);
  }, []);

  // Fetch trades from backend
  const fetchTrades = useCallback(async () => {
    if (isFetchingTrades.current) return; // Prevent concurrent fetches
    isFetchingTrades.current = true;

    try {
      const res = await axios.get("/latest-trades");
      if (res.data.trades) {
        const now = Math.floor(Date.now() / 1000);
        const updatedTrades = res.data.trades.map(t => {
          const created = Math.floor(new Date(t.created_at).getTime() / 1000);
          const expiryTimestamp = Math.floor(new Date(t.expiry_at).getTime() / 1000);
          let timeLeft = expiryTimestamp - now;
          const type = t.direction?.toString().toUpperCase() || "N/A";

          // Map crypto field to asset field for consistency
          const asset = t.asset || t.crypto;

          // Live price logic for active trades
          let currentPrice = t.purchase_price;
          if (marketPrices[asset]?.price) {
            currentPrice = parseFloat(marketPrices[asset].price.replace(/,/g, ''));
          }

          // Prioritize profit from backend if set
          let profit = t.profit !== null ? Number(t.profit) : null;
          let delivery_price = t.delivery_price || null;

          if (timeLeft <= 0) {
            // Trade finished
            const levFactor = Number(t.leverage) || 1;

            // If backend hasn't provided a profit yet, calculate it locally
            if (profit === null) {
              if (t.status === "approved") {
                profit = (Number(t.amount) * levFactor) / 100;
                delivery_price = t.delivery_price || currentPrice || t.purchase_price;
              } else if (t.status === "declined") {
                profit = -((Number(t.amount) * levFactor) / 100);
                delivery_price = t.delivery_price || currentPrice || t.purchase_price;
              } else {
                profit = -((Number(t.amount) * levFactor) / 100);
                delivery_price = currentPrice || t.purchase_price;
              }
            }
            timeLeft = 0;
          } else {
            // Trade active - show live price as delivery price
            delivery_price = currentPrice || t.purchase_price;
          }

          return {
            ...t,
            asset, // Ensure asset field is always present
            type,
            timeLeft: Math.max(timeLeft, 0),
            profit,
            delivery_price,
          };
        });

        // Always sort by ID DESC (newest first)
        const sortedUpdatedTrades = updatedTrades.sort((a, b) => b.id - a.id);

        // Only update if trades actually changed to prevent unnecessary re-renders
        setTrades(prevTrades => {
          // If we have the same number of trades and same IDs, preserve timeLeft from state
          if (prevTrades.length === sortedUpdatedTrades.length) {
            return sortedUpdatedTrades.map(newTrade => {
              const existingTrade = prevTrades.find(t => t.id === newTrade.id);
              if (existingTrade && existingTrade.timeLeft > 0 && newTrade.timeLeft > 0) {
                // Preserve the countdown timer state for active trades
                return { ...newTrade, timeLeft: existingTrade.timeLeft };
              }
              return newTrade;
            });
          }
          return sortedUpdatedTrades;
        });
      }
    } catch (err) {
      if (err.response?.status === 401) {
        window.location.href = "/login";
        return;
      }
      console.error("Failed to fetch trades", err);
    } finally {
      isFetchingTrades.current = false;
    }
  }, [marketPrices]);

  useEffect(() => {
    fetchTrades();
    const interval = setInterval(fetchTrades, 3000);
    return () => clearInterval(interval);
  }, [fetchTrades]);

  // Timer countdown
  useEffect(() => {
    const interval = setInterval(() => {
      setTrades(prev =>
        prev.map(t => {
          if (t.timeLeft > 0) {
            const newTimeLeft = t.timeLeft - 1;
            if (newTimeLeft <= 0) {
              const levFactor = Number(t.leverage) || 1;
              // Use leverage formula for both approved and pending/declined
              let profit = t.status === "approved"
                ? (Number(t.amount) * levFactor) / 100
                : -((Number(t.amount) * levFactor) / 100);

              // Optimistic update removed - relying on backend synchronization
              // setWallet(prevWallet => { ... });

              return { ...t, timeLeft: 0, profit, delivery_price: t.delivery_price || t.purchase_price };
            }
            return { ...t, timeLeft: newTimeLeft };
          }
          return t;
        })
      );
    }, 1000);
    return () => clearInterval(interval);
  }, []);

  // Auto-select expiry when leverage changes
  useEffect(() => {
    if (leverage && leverageExpiryMap[leverage]) {
      setExpiry(leverageExpiryMap[leverage]);
    }
  }, [leverage]);

  const placeTrade = async type => {
    setError("");
    const numericAmount = parseFloat(amount);
    if (!numericAmount || numericAmount <= 0) return setError("Enter a valid amount.");
    // Remove balance check since we're not deducting on placement anymore
    // if (!wallet[selectedCrypto] || numericAmount > wallet[selectedCrypto]) return setError("Insufficient balance!");

    if (!leverage) return setError("Select leverage");
    if (!expiry) return setError("Select expiry");

    // Calculate total USD balance from all wallet balances
    const totalUsdBalance = Object.keys(wallet).reduce((total, key) => {
      return total + (Number(wallet[key]) || 0);
    }, 0);

    // Initial check for specific coin balance
    const availableCoinBalance = Number(wallet[selectedCrypto]) || 0;
    if (numericAmount > availableCoinBalance) {
      setPopupMessage(`Insufficient balance! You attempted to trade ${numericAmount} ${selectedCrypto}, but your available balance in ${selectedCrypto} is only ${availableCoinBalance} ${selectedCrypto}.`);
      setShowPopup(true);
      return;
    }

    // Balance validation based on expiry
    if (expiry === 120) { // 2 mins
      if (totalUsdBalance < 5000) {
        setPopupMessage(`You are not eligible for this trade. 2-minute trades require a minimum balance of $5,000 across all wallets. Your current total is $${totalUsdBalance.toLocaleString()}.`);
        setShowPopup(true);
        return;
      }
    } else if (expiry === 43200) { // 12 hours
      if (totalUsdBalance < 40000) {
        setPopupMessage(`You are not eligible for this trade. 12-hour trades require a minimum balance of $40,000 across all wallets. Your current total is $${totalUsdBalance.toLocaleString()}.`);
        setShowPopup(true);
        return;
      }
    } else if (expiry === 86400 || expiry === 259200 || expiry === 604800) { // 1, 3, 7 days
      if (totalUsdBalance < 80000) {
        setPopupMessage(`You are not eligible for this trade. 1, 3, and 7-day trades require a minimum balance of $80,000 across all wallets. Your current total is $${totalUsdBalance.toLocaleString()}.`);
        setShowPopup(true);
        return;
      }
    }

    if (type === "buy") setLoadingBuy(true);
    else setLoadingSell(true);
    try {
      const res = await axios.post("/trade", {
        crypto: selectedCrypto,
        amount: numericAmount,
        leverage,
        expiry: expiry.toString(),
        type,
      });

      // Don't add trade optimistically - let fetchTrades handle it
      // This prevents duplicate trades from appearing

      // Optimistic wallet update for "instant" balance change
      setWallet(prev => ({
        ...prev,
        [selectedCrypto]: (Number(prev[selectedCrypto]) || 0) - numericAmount
      }));

      setAmount("");

      // Fetch trades immediately to show the new trade
      await fetchTrades();
    } catch (err) {
      const errorMsg = err.response?.data?.message || err.response?.data?.errors?.amount?.[0] || "Trade failed!";
      setError(errorMsg);
      setPopupMessage(errorMsg);
      setShowPopup(true);
    } finally {
      setLoadingBuy(false);
      setLoadingSell(false);
    }
  };

  // Utility function for formatting time
  const formatTime = (seconds) => {
    const d = Math.floor(seconds / 86400);
    const h = Math.floor((seconds % 86400) / 3600);
    const m = Math.floor((seconds % 3600) / 60);
    const s = seconds % 60;

    if (d > 0) return `${d}d ${h}h ${m}m`;
    if (h > 0) return `${h}h ${m}m ${s}s`;
    return `${m.toString().padStart(2, "0")}:${s.toString().padStart(2, "0")}`;
  };

  const totalProfit = trades.filter(t => t.profit !== null && t.asset === selectedCrypto).reduce((sum, t) => sum + Number(t.profit), 0);
  const toLocalTime = ts => (ts ? new Date(ts).toLocaleString() : "Pending");

  return (
    <div className="min-h-screen bg-gray-100 dark:bg-gray-900 text-gray-900 dark:text-gray-100">
      <Navbar user={user} />
      <div className="p-4 md:p-6 grid grid-cols-12 gap-4 md:gap-6">

        {/* MOBILE: Balance Overview - Shows first on mobile */}
        <div className="md:hidden col-span-12 order-1 relative z-10">
          <div className="bg-white dark:bg-gray-800 p-5 rounded-xl shadow-lg hover:shadow-2xl transition-shadow duration-300">
            <h3 className="text-lg font-bold mb-4">{selectedCrypto} Overview</h3>
            <p>Balance: {Number(wallet[selectedCrypto] || 0).toFixed(2)}</p>
            <p className={`font-medium ${totalProfit >= 0 ? "text-green-500" : "text-red-500"}`}>
              Total Profit: {Number(totalProfit || 0).toFixed(2)}
            </p>
          </div>
        </div>

        {/* LEFT - Market Prices */}
        <div className="hidden md:block col-span-12 md:col-span-3 lg:col-span-2 relative z-10">
          <div className="bg-gradient-to-br from-white to-gray-50 dark:from-gray-800 dark:to-gray-900 rounded-2xl shadow-2xl p-4 h-[calc(100vh-120px)] hover:shadow-3xl transition-shadow duration-300">
            <h3 className="text-lg font-bold mb-4 bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">Live Markets</h3>
            <div className="grid grid-cols-1 gap-2 h-[calc(100%-3rem)]">
              {marketList.slice(0, typeof window !== 'undefined' ? Math.floor((window.innerHeight - 200) / 70) : marketList.length).map((coin, index) => {
                const data = marketPrices[coin] || {};
                const isSelected = selectedCrypto === coin;
                const isPositive = parseFloat(data.change) >= 0;

                return (
                  <div
                    key={coin}
                    onClick={() => setSelectedCrypto(coin)}
                    style={{ animationDelay: `${index * 50}ms` }}
                    className={`cursor-pointer flex items-center justify-between p-3 rounded-xl transition-all duration-300 animate-fade-in ${isSelected
                      ? "bg-gradient-to-r from-blue-500 to-purple-600 text-white shadow-lg scale-105"
                      : "bg-white/60 dark:bg-gray-700/50 hover:bg-white dark:hover:bg-gray-700 hover:shadow-md hover:scale-102"
                      }`}
                  >
                    <div className="flex items-center gap-3">
                      {data.image ? (
                        <img src={data.image} alt={coin} className="w-10 h-10 rounded-full shadow-md" />
                      ) : (
                        <div className="w-10 h-10 rounded-full bg-gradient-to-br from-gray-200 to-gray-300 dark:from-gray-600 dark:to-gray-700 animate-pulse" />
                      )}
                      <div>
                        <div className={`font-bold text-sm ${isSelected ? "text-white" : "text-gray-900 dark:text-gray-100"}`}>{coin}</div>
                        <div className={`text-xs ${isSelected ? "text-white/80" : "text-gray-500 dark:text-gray-400"}`}>USDT</div>
                      </div>
                    </div>
                    <div className="text-right">
                      <div className={`font-semibold text-sm ${isSelected ? "text-white" : "text-gray-900 dark:text-gray-100"}`}>
                        {data.price ? `$${data.price}` : <span className="animate-pulse">...</span>}
                      </div>
                      {data.change && (
                        <div className={`text-xs font-medium px-2 py-0.5 rounded-full ${isPositive
                          ? isSelected ? "bg-green-400/30 text-white" : "bg-green-100 text-green-600 dark:bg-green-900/30 dark:text-green-400"
                          : isSelected ? "bg-red-400/30 text-white" : "bg-red-100 text-red-600 dark:bg-red-900/30 dark:text-red-400"
                          }`}>
                          {isPositive ? "+" : ""}{data.change}%
                        </div>
                      )}
                    </div>
                  </div>
                );
              })}
            </div>
          </div>
        </div>

        {/* CENTER - Main Chart + Trades */}
        <div className="col-span-12 md:col-span-6 lg:col-span-7 space-y-6 order-2 md:order-2 relative z-10">
          <TradingChart symbol={selectedCrypto} theme={theme} />

          {/* MOBILE: Trade Asset Section - Shows third on mobile */}
          <div className="md:hidden relative z-10">
            <div className="bg-white dark:bg-gray-800 p-5 rounded-xl shadow-lg hover:shadow-2xl transition-shadow duration-300">
              <h3 className="text-lg font-bold mb-4">Trade Asset</h3>
              <div className="space-y-4">
                <div>
                  <label className="text-xs text-gray-500 mb-1 block">Select Crypto Coin</label>
                  <select
                    value={selectedCrypto}
                    onChange={e => setSelectedCrypto(e.target.value)}
                    className="w-full p-3 rounded-lg bg-gray-100 dark:bg-gray-700 border-none focus:ring-2 focus:ring-blue-500 transition-all"
                  >
                    <option value="" disabled>Select Crypto Coin</option>
                    {marketList.map(c => <option key={c} value={c}>{c}</option>)}
                  </select>
                </div>
                <div>
                  <label className="text-xs text-gray-500 mb-1 block">Amount</label>
                  <div className="flex items-center bg-gray-100 dark:bg-gray-700 rounded-lg overflow-hidden">
                    <input
                      type="number"
                      value={amount}
                      placeholder="Enter USDT"
                      onChange={e => setAmount(e.target.value)}
                      className="w-full p-3 bg-transparent border-none focus:ring-0"
                    />
                    <div className="px-4 py-3 bg-gray-200 dark:bg-gray-600 text-sm font-semibold text-gray-600 dark:text-gray-300">USDT</div>
                  </div>
                </div>
                <div>
                  <label className="text-xs text-gray-500 mb-1 block">Leverage</label>
                  <select
                    value={leverage}
                    onChange={e => setLeverage(e.target.value)}
                    className="w-full p-3 rounded-lg bg-gray-100 dark:bg-gray-700 border-none focus:ring-2 focus:ring-blue-500 transition-all"
                  >
                    <option value="" disabled>Leverage</option>
                    <option>1:10</option>
                    <option>1:35</option>
                    <option>1:87</option>
                    <option>1:165</option>
                    <option>1:250</option>
                    <option>1:365</option>
                  </select>
                </div>
                <div>
                  <label className="text-xs text-gray-500 mb-1 block">Expiry</label>
                  <select
                    value={expiry}
                    onChange={e => setExpiry(Number(e.target.value))}
                    className="w-full p-3 rounded-lg bg-gray-100 dark:bg-gray-700 border-none focus:ring-2 focus:ring-blue-500 transition-all"
                  >
                    <option value="" disabled>Expiry</option>
                    {expiryOptions.map(opt => <option key={opt.value} value={opt.value}>{opt.label}</option>)}
                  </select>
                </div>
                {error && <p className="text-red-500 text-sm animate-pulse">{error}</p>}
                <div className="flex gap-3 pt-2">
                  <button
                    onClick={() => placeTrade("buy")}
                    disabled={loadingBuy || loadingSell}
                    className={`flex-1 py-3 rounded-lg font-bold text-white transition-all duration-300 transform hover:scale-105 active:scale-95 ${loadingBuy
                      ? "bg-green-600 opacity-70 cursor-not-allowed"
                      : "bg-gradient-to-r from-green-500 to-green-600 hover:from-green-600 hover:to-green-700 shadow-md"
                      }`}
                  >
                    {loadingBuy ? (
                      <span className="flex items-center justify-center gap-2">
                        <svg className="animate-spin h-5 w-5" viewBox="0 0 24 24">
                          <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none" />
                          <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
                        </svg>
                        Processing...
                      </span>
                    ) : "Buy"}
                  </button>
                  <button
                    onClick={() => placeTrade("sell")}
                    disabled={loadingBuy || loadingSell}
                    className={`flex-1 py-3 rounded-lg font-bold text-white transition-all duration-300 transform hover:scale-105 active:scale-95 ${loadingSell
                      ? "bg-red-600 opacity-70 cursor-not-allowed"
                      : "bg-gradient-to-r from-red-500 to-red-600 hover:from-red-600 hover:to-red-700 shadow-md"
                      }`}
                  >
                    {loadingSell ? (
                      <span className="flex items-center justify-center gap-2">
                        <svg className="animate-spin h-5 w-5" viewBox="0 0 24 24">
                          <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none" />
                          <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
                        </svg>
                        Processing...
                      </span>
                    ) : "Sell"}
                  </button>
                </div>
              </div>
            </div>
          </div>

          {/* Trades Table - Shows fourth on mobile */}
          <div className="order-4 md:order-3 relative z-10">
            <div className="bg-white dark:bg-gray-800 p-4 rounded-xl shadow-lg hover:shadow-2xl transition-shadow duration-300 overflow-x-auto">
              <h3 className="text-xl font-bold mb-3">Latest Trades</h3>
              <table className="w-full table-auto">
                <thead>
                  <tr className="border-b dark:border-gray-700">
                    <th className="p-2 text-left">Details</th>
                    <th className="p-2 text-center">Total Amount</th>
                    <th className="p-2 text-center">
                      {trades.length > 0 && trades[0].timeLeft > 0
                        ? "Duration"
                        : "Net Amount"}
                    </th>
                    <th className="p-2 text-center">Status</th>
                    <th className="p-2 text-center">Delivery Date</th>
                    <th className="p-2 text-center"></th>
                  </tr>
                </thead>
                <tbody>
                  {trades.length === 0 && (
                    <tr>
                      <td colSpan="6" className="p-4 text-center text-gray-500">No trades yet.</td>
                    </tr>
                  )}
                  {trades.map(trade => {
                    const isOpen = expandedTrade === trade.id;
                    return (
                      <React.Fragment key={trade.id}>
                        <tr className="border-b dark:border-gray-700 hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors">
                          <td className="p-3 text-left">
                            <span className="text-green-500 font-semibold">↑ {trade.asset}/USDT</span>
                            <br />
                            <span className="text-xs opacity-70">{toLocalTime(trade.created_at)}</span>
                          </td>
                          <td className="p-3 text-center font-semibold">${Number(trade.amount).toFixed(2)}</td>
                          <td className="p-3 text-center font-semibold text-lg">
                            {trade.timeLeft > 0
                              ? <span className="text-blue-500">{formatTime(trade.timeLeft)}</span>
                              : trade.profit !== null
                                ? trade.profit > 0
                                  ? <span className="text-green-500">${Math.abs(Number(trade.profit)).toFixed(2)}</span>
                                  : <span className="text-red-500">${Math.abs(Number(trade.profit)).toFixed(2)}</span>
                                : <span className="text-gray-500">Pending</span>
                            }
                          </td>
                          <td className="p-3 text-center">
                            <span className={`px-3 py-1 rounded-full text-white text-xs font-medium ${trade.type === "BUY" ? "bg-green-500" : trade.type === "SELL" ? "bg-red-500" : "bg-gray-400"}`}>
                              {trade.type}
                            </span>
                          </td>
                          <td className="p-3 text-center text-xs opacity-70">
                            {toLocalTime(trade.created_at)}
                          </td>
                          <td className="p-3 text-center">
                            <button
                              onClick={() => setExpandedTrade(isOpen ? null : trade.id)}
                              className="text-blue-500 hover:text-blue-700 text-sm font-medium transition-colors"
                            >
                              {isOpen ? "Hide ▲" : "Details ▼"}
                            </button>
                          </td>
                        </tr>
                        <AnimatePresence>
                          {isOpen && (
                            <tr>
                              <td colSpan="6" className="p-0 overflow-hidden">
                                <motion.div
                                  initial={{ height: 0, opacity: 0 }}
                                  animate={{ height: "auto", opacity: 1 }}
                                  exit={{ height: 0, opacity: 0 }}
                                  transition={{ duration: 0.4, ease: "easeInOut" }}
                                >
                                  <div className="bg-gradient-to-r from-gray-50 to-gray-100 dark:from-gray-900 dark:to-gray-800 p-6 border-b dark:border-gray-700">
                                    <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                                      <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow-sm border dark:border-gray-700">
                                        <p className="text-xs text-gray-500 dark:text-gray-400 mb-1">Total Amount</p>
                                        <p className="text-lg font-bold text-gray-900 dark:text-gray-100">${Number(trade.amount).toFixed(2)}</p>
                                      </div>
                                      <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow-sm border dark:border-gray-700">
                                        <p className="text-xs text-gray-500 dark:text-gray-400 mb-1">Purchase Price</p>
                                        <p className="text-lg font-bold text-gray-900 dark:text-gray-100">
                                          {trade.purchase_price ? `$${Number(trade.purchase_price).toFixed(2)}` : "Loading..."}
                                        </p>
                                      </div>
                                      <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow-sm border dark:border-gray-700">
                                        <p className="text-xs text-gray-500 dark:text-gray-400 mb-1">Delivery Price</p>
                                        <p className="text-lg font-bold text-gray-900 dark:text-gray-100">
                                          {trade.delivery_price ? `$${Number(trade.delivery_price).toFixed(2)}` : "Loading..."}
                                        </p>
                                      </div>
                                      <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow-sm border dark:border-gray-700">
                                        <p className="text-xs text-gray-500 dark:text-gray-400 mb-1">Leverage</p>
                                        <p className="text-lg font-bold text-gray-900 dark:text-gray-100">1:{Math.floor(trade.leverage)}</p>
                                      </div>
                                      {trade.profit !== null && (
                                        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow-sm border dark:border-gray-700">
                                          <p className="text-xs text-gray-500 dark:text-gray-400 mb-1">
                                            {trade.profit > 0 ? "Net Profit Amount" : "Net Loss Amount"}
                                          </p>
                                          <p className={`text-lg font-bold ${trade.profit > 0 ? "text-green-500" : "text-red-500"}`}>
                                            ${Math.abs(Number(trade.profit)).toFixed(2)}
                                          </p>
                                        </div>
                                      )}
                                      <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow-sm border dark:border-gray-700">
                                        <p className="text-xs text-gray-500 dark:text-gray-400 mb-1">Delivery Date</p>
                                        <p className="text-sm font-semibold text-gray-900 dark:text-gray-100">{toLocalTime(trade.created_at)}</p>
                                      </div>
                                    </div>
                                  </div>
                                </motion.div>
                              </td>
                            </tr>
                          )}
                        </AnimatePresence>
                      </React.Fragment>
                    );
                  })}
                </tbody>
              </table>
            </div>
          </div>
        </div>

        {/* RIGHT - Wallet + Trade Panel - Desktop only */}
        <div className="hidden md:block col-span-12 md:col-span-3 space-y-6 order-3 relative z-10">
          <div className="bg-white dark:bg-gray-800 p-5 rounded-xl shadow-lg hover:shadow-2xl transition-shadow duration-300">
            <h3 className="text-lg font-bold mb-4">{selectedCrypto} Overview</h3>
            <p>Balance: {Number(wallet[selectedCrypto] || 0).toFixed(2)}</p>
            <p className={`font-medium ${totalProfit >= 0 ? "text-green-500" : "text-red-500"}`}>
              Total Profit: {Number(totalProfit || 0).toFixed(2)}
            </p>
          </div>

          <div className="bg-white dark:bg-gray-800 p-5 rounded-xl shadow-lg hover:shadow-2xl transition-shadow duration-300">
            <h3 className="text-lg font-bold mb-4">Trade Asset</h3>
            <div className="space-y-4">

              {/* Crypto Select */}
              <div>
                <label className="text-xs text-gray-500 mb-1 block">Select Crypto Coin</label>
                <select
                  value={selectedCrypto}
                  onChange={e => setSelectedCrypto(e.target.value)}
                  className="w-full p-3 rounded-lg bg-gray-100 dark:bg-gray-700 border-none focus:ring-2 focus:ring-blue-500 transition-all"
                >
                  <option value="" disabled>Select Crypto Coin</option>
                  {marketList.map(c => <option key={c} value={c}>{c}</option>)}
                </select>
              </div>

              {/* Amount Input */}
              <div>
                <label className="text-xs text-gray-500 mb-1 block">Amount</label>
                <div className="flex items-center bg-gray-100 dark:bg-gray-700 rounded-lg overflow-hidden">
                  <input
                    type="number"
                    value={amount}
                    placeholder="Enter USDT"
                    onChange={e => setAmount(e.target.value)}
                    className="w-full p-3 bg-transparent border-none focus:ring-0"
                  />
                  <div className="px-4 py-3 bg-gray-200 dark:bg-gray-600 text-sm font-semibold text-gray-600 dark:text-gray-300">
                    USDT
                  </div>
                </div>
              </div>

              {/* Leverage Select */}
              <div>
                <label className="text-xs text-gray-500 mb-1 block">Leverage</label>
                <select
                  value={leverage}
                  onChange={e => setLeverage(e.target.value)}
                  className="w-full p-3 rounded-lg bg-gray-100 dark:bg-gray-700 border-none focus:ring-2 focus:ring-blue-500 transition-all"
                >
                  <option value="" disabled>Leverage</option>
                  <option>1:10</option>
                  <option>1:35</option>
                  <option>1:87</option>
                  <option>1:165</option>
                  <option>1:250</option>
                  <option>1:365</option>
                </select>
              </div>

              {/* Expiry Select */}
              <div>
                <label className="text-xs text-gray-500 mb-1 block">Expiry</label>
                <select
                  value={expiry}
                  onChange={e => setExpiry(Number(e.target.value))}
                  className="w-full p-3 rounded-lg bg-gray-100 dark:bg-gray-700 border-none focus:ring-2 focus:ring-blue-500 transition-all"
                >
                  <option value="" disabled>Expiry</option>
                  {expiryOptions.map(opt => <option key={opt.value} value={opt.value}>{opt.label}</option>)}
                </select>
              </div>

              {error && <p className="text-red-500 text-sm animate-pulse">{error}</p>}

              <div className="flex gap-3 pt-2">
                <button
                  onClick={() => placeTrade("buy")}
                  disabled={loadingBuy || loadingSell}
                  className={`flex-1 py-3 rounded-lg font-bold text-white transition-all duration-300 transform hover:scale-105 active:scale-95 ${loadingBuy
                    ? "bg-green-600 opacity-70 cursor-not-allowed"
                    : "bg-gradient-to-r from-green-500 to-green-600 hover:from-green-600 hover:to-green-700 shadow-md"
                    }`}
                >
                  {loadingBuy ? (
                    <span className="flex items-center justify-center gap-2">
                      <svg className="animate-spin h-5 w-5" viewBox="0 0 24 24">
                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none" />
                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
                      </svg>
                      Processing...
                    </span>
                  ) : "Buy"}
                </button>
                <button
                  onClick={() => placeTrade("sell")}
                  disabled={loadingBuy || loadingSell}
                  className={`flex-1 py-3 rounded-lg font-bold text-white transition-all duration-300 transform hover:scale-105 active:scale-95 ${loadingSell
                    ? "bg-red-600 opacity-70 cursor-not-allowed"
                    : "bg-gradient-to-r from-red-500 to-red-600 hover:from-red-600 hover:to-red-700 shadow-md"
                    }`}
                >
                  {loadingSell ? (
                    <span className="flex items-center justify-center gap-2">
                      <svg className="animate-spin h-5 w-5" viewBox="0 0 24 24">
                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none" />
                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
                      </svg>
                      Processing...
                    </span>
                  ) : "Sell"}
                </button>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Popup Modal */}
      {showPopup && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-2xl p-8 max-w-md mx-4 shadow-2xl animate-fade-in">
            <div className="text-center">
              <div className="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-red-100 dark:bg-red-900/30 mb-4">
                <svg className="h-6 w-6 text-red-600 dark:text-red-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
                </svg>
              </div>
              <h3 className="text-lg font-bold text-gray-900 dark:text-gray-100 mb-2">Trade Not Eligible</h3>
              <p className="text-gray-600 dark:text-gray-400 mb-6">{popupMessage}</p>
              <button
                onClick={() => setShowPopup(false)}
                className="w-full py-3 px-4 bg-gradient-to-r from-blue-500 to-blue-600 hover:from-blue-600 hover:to-blue-700 text-white font-semibold rounded-lg transition-all duration-300 transform hover:scale-105 active:scale-95 shadow-md"
              >
                Understood
              </button>
            </div>
          </div>
        </div>
      )}

      <Styles />
    </div>
  );
}

// Styles
const Styles = () => (
  <style>
    {`
      .scrollbar-hide::-webkit-scrollbar { display: none; }
      .scrollbar-hide { -ms-overflow-style: none; scrollbar-width: none; }
      
      @keyframes fade-in {
        from { opacity: 0; transform: translateY(10px); }
        to { opacity: 1; transform: translateY(0); }
      }
      
      .animate-fade-in {
        animation: fade-in 0.5s ease-out forwards;
      }
      
      .hover\:scale-102:hover {
        transform: scale(1.02);
      }
      
      .custom-scrollbar::-webkit-scrollbar { width: 6px; }
      .custom-scrollbar::-webkit-scrollbar-track { background: transparent; }
      .custom-scrollbar::-webkit-scrollbar-thumb { background-color: rgba(156, 163, 175, 0.5); border-radius: 20px; }
      .custom-scrollbar::-webkit-scrollbar-thumb:hover { background-color: rgba(107, 114, 128, 0.8); }
    `}
  </style>
);
