import { useState, useRef, useEffect } from 'react';
import { useForm, Head, Link } from '@inertiajs/react';
import { FaBitcoin, FaEthereum, FaCoins } from 'react-icons/fa';

export default function VerifyEmail({ email, message }) {
    const [code, setCode] = useState(['', '', '', '', '', '']);
    const inputRefs = [useRef(), useRef(), useRef(), useRef(), useRef(), useRef()];
    const { data, setData, post, processing, errors } = useForm({
        email: email || '',
        code: '',
    });

    const handleChange = (index, value) => {
        if (value.length > 1) value = value[0];
        if (!/^\d*$/.test(value)) return;

        const newCode = [...code];
        newCode[index] = value;
        setCode(newCode);

        // Auto-focus next input
        if (value && index < 5) {
            inputRefs[index + 1].current?.focus();
        }

        // Update form data
        setData('code', newCode.join(''));
    };

    const handleKeyDown = (index, e) => {
        if (e.key === 'Backspace' && !code[index] && index > 0) {
            inputRefs[index - 1].current?.focus();
        }
    };

    const handlePaste = (e) => {
        e.preventDefault();
        const pastedData = e.clipboardData.getData('text').slice(0, 6);
        if (!/^\d+$/.test(pastedData)) return;

        const newCode = pastedData.split('');
        while (newCode.length < 6) newCode.push('');
        setCode(newCode);
        setData('code', newCode.join(''));
        inputRefs[5].current?.focus();
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route('verification.code.verify'));
    };

    const handleResend = () => {
        post(route('verification.code.resend'), {
            preserveScroll: true,
        });
    };

    return (
        <div className="min-h-screen flex">
            <Head title="Verify Email" />

            {/* Left Branding Panel */}
            <div className="hidden lg:flex flex-1 relative items-center justify-center 
                bg-gradient-to-r from-purple-600 via-blue-600 to-indigo-600 text-white shadow-lg overflow-hidden">
                <div className="absolute w-96 h-96 top-[-50px] left-[-80px] bg-blue-500 rounded-full blur-[140px] opacity-30"></div>
                <div className="absolute w-[500px] h-[500px] bottom-[-100px] right-[-150px] bg-purple-500 rounded-full blur-[180px] opacity-25"></div>

                <FaBitcoin className="absolute text-yellow-400 text-7xl animate-float-slow top-32 left-24 opacity-90" />
                <FaEthereum className="absolute text-gray-300 text-6xl animate-float-fast top-56 left-72 opacity-90" />
                <FaCoins className="absolute text-orange-400 text-7xl animate-float-slower top-80 left-14 opacity-90" />

                <div className="relative z-10 text-center px-10">
                    <h1 className="text-white text-6xl font-black tracking-wide">CoinSova</h1>
                    <p className="text-white-400 text-lg mt-5 leading-relaxed max-w-md">
                        Execute trades fast, securely and with deep liquidity
                    </p>
                    <div className="mx-auto mt-8 w-32 h-1 bg-gradient-to-r from-orange-400 to-yellow-500 rounded-full"></div>
                </div>
            </div>

            {/* Right form */}
            <div className="flex flex-1 items-center justify-center bg-white dark:bg-gray-900 w-full h-screen p-10 sm:p-12">
                <div className="w-full max-w-md lg:max-w-lg">
                    <div className="text-center mb-8">
                        <div className="w-20 h-20 bg-gradient-to-r from-purple-600 to-blue-600 rounded-full flex items-center justify-center mx-auto mb-4">
                            <svg className="w-10 h-10 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
                            </svg>
                        </div>
                        <h2 className="text-3xl font-extrabold text-gray-900 dark:text-white mb-2">
                            Verify Your Email
                        </h2>
                        <p className="text-sm text-gray-600 dark:text-gray-400">
                            We've sent a 6-digit code to<br />
                            <span className="font-semibold text-blue-600 dark:text-blue-400">{email}</span>
                        </p>
                    </div>

                    {message && (
                        <div className="mb-6 p-4 rounded-xl bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800">
                            <p className="text-sm text-green-600 dark:text-green-400 text-center">{message}</p>
                        </div>
                    )}

                    {errors.code && (
                        <div className="mb-6 p-4 rounded-xl bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800">
                            <p className="text-sm text-red-600 dark:text-red-400 text-center">{errors.code}</p>
                        </div>
                    )}

                    <form onSubmit={handleSubmit} className="space-y-8">
                        <div>
                            <label className="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-4 text-center">
                                Enter Verification Code
                            </label>
                            <div className="flex justify-center gap-3" onPaste={handlePaste}>
                                {code.map((digit, index) => (
                                    <input
                                        key={index}
                                        ref={inputRefs[index]}
                                        type="text"
                                        inputMode="numeric"
                                        maxLength={1}
                                        value={digit}
                                        onChange={(e) => handleChange(index, e.target.value)}
                                        onKeyDown={(e) => handleKeyDown(index, e)}
                                        className="w-14 h-16 text-center text-2xl font-bold border-2 border-gray-300 dark:border-gray-600 rounded-xl focus:border-purple-500 focus:ring-2 focus:ring-purple-400 dark:bg-gray-800 dark:text-white transition-all"
                                    />
                                ))}
                            </div>
                        </div>

                        <button
                            type="submit"
                            disabled={processing || code.join('').length !== 6}
                            className="w-full py-4 bg-gradient-to-r from-purple-600 via-blue-600 to-indigo-600 text-white font-bold text-lg rounded-xl shadow-2xl hover:shadow-purple-500/50 transform hover:scale-105 hover:-translate-y-1 transition-all duration-300 disabled:opacity-50 disabled:cursor-not-allowed relative overflow-hidden group"
                        >
                            <span className="relative z-10">{processing ? 'Verifying...' : 'Verify Email'}</span>
                            <div className="absolute inset-0 bg-gradient-to-r from-purple-500 via-blue-500 to-indigo-500 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                        </button>
                    </form>

                    <div className="mt-6 text-center space-y-3">
                        <p className="text-sm text-gray-600 dark:text-gray-400">
                            Didn't receive the code?
                        </p>
                        <button
                            onClick={handleResend}
                            disabled={processing}
                            className="text-sm text-blue-600 hover:text-blue-700 dark:text-blue-400 dark:hover:text-blue-300 font-semibold transition-colors disabled:opacity-50"
                        >
                            Resend Code
                        </button>
                        <div>
                            <Link
                                href={route('login')}
                                className="text-sm text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 transition-colors"
                            >
                                ← Back to Login
                            </Link>
                        </div>
                    </div>
                </div>
            </div>

            <style>{`
                @keyframes float-slow {
                    0%, 100% { transform: translateY(0px); }
                    50% { transform: translateY(-20px); }
                }
                @keyframes float-slower {
                    0%, 100% { transform: translateY(0px); }
                    50% { transform: translateY(-10px); }
                }
                @keyframes float-fast {
                    0%, 100% { transform: translateY(0px); }
                    50% { transform: translateY(-30px); }
                }
                .animate-float-slow { animation: float-slow 5s ease-in-out infinite; }
                .animate-float-slower { animation: float-slower 6s ease-in-out infinite; }
                .animate-float-fast { animation: float-fast 4s ease-in-out infinite; }
            `}</style>
        </div>
    );
}
