import { useEffect, useState } from 'react';
import InputError from '@/Components/InputError';
import InputLabel from '@/Components/InputLabel';
import PrimaryButton from '@/Components/PrimaryButton';
import TextInput from '@/Components/TextInput';
import { Head, Link, useForm } from '@inertiajs/react';
import { motion } from "framer-motion";
import { FaBitcoin, FaEthereum, FaDollarSign, FaCoins } from 'react-icons/fa';

export default function Register() {
    const [showPassword, setShowPassword] = useState(false);
    const [showConfirmPassword, setShowConfirmPassword] = useState(false);
    const { data, setData, post, processing, errors, reset } = useForm({
        name: '',
        email: '',
        password: '',
        password_confirmation: '',
        device_timezone: '',
        device_datetime: '',
    });

    useEffect(() => {
        const timezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
        const localDatetime = new Date().toISOString();

        setData('device_timezone', timezone);
        setData('device_datetime', localDatetime);

        return () => {
            reset('password', 'password_confirmation');
        };
    }, []);

    const submit = (e) => {
        e.preventDefault();
        post(route('register'));
    };

    return (
        <div className="min-h-screen flex">
            <Head title="Register" />

            {/* Left Branding Panel */}
            <div className="hidden lg:flex flex-1 relative items-center justify-center bg-gradient-to-r from-purple-600 via-blue-600 to-indigo-600 text-white shadow-lg overflow-hidden">
                {/* Background glow blobs */}
                <div className="absolute w-96 h-96 top-[-50px] left-[-80px] bg-orange-500 rounded-full blur-[140px] opacity-30"></div>
                <div className="absolute w-[500px] h-[500px] bottom-[-100px] right-[-150px] bg-yellow-500 rounded-full blur-[180px] opacity-25"></div>

                {/* Floating crypto icons */}
                <FaBitcoin className="absolute text-yellow-400 text-7xl animate-float-slow top-32 left-24 opacity-90" />
                <FaEthereum className="absolute text-gray-300 text-6xl animate-float-fast top-56 left-72 opacity-90" />
                <FaCoins className="absolute text-orange-400 text-7xl animate-float-slower top-80 left-14 opacity-90" />

                {/* Branding text */}
                <div className="relative z-10 text-center px-10">
                    <h1 className="text-white text-6xl font-black tracking-wide mb-4">
                        CryptoTrade
                    </h1>
                    <p className="text-white-300 text-lg leading-relaxed max-w-md mx-auto">
                        Execute trades fast, securely and with deep liquidity — built for professional traders.
                    </p>
                    <div className="mx-auto mt-8 w-32 h-1 bg-gradient-to-r from-orange-400 to-yellow-500 rounded-full"></div>
                </div>
            </div>

            {/* Right Register Form */}
            <div className="flex flex-1 items-center justify-center bg-white w-full h-screen p-10 sm:p-12">
                <motion.div
                    initial={{ opacity: 0, y: 25 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.5 }}
                    className="w-full max-w-md bg-white shadow-xl rounded-2xl p-10 border border-gray-100"
                >
                    <h1 className="text-3xl font-extrabold text-center mb-2 text-gray-900 tracking-tight">
                        Create Account
                    </h1>
                    <p className="text-center text-gray-500 mb-6 text-sm">
                        Join and start trading instantly
                    </p>

                    <form onSubmit={submit} className="space-y-5">
                        {/* NAME */}
                        <div>
                            <InputLabel htmlFor="name" value="Name" className="font-medium text-gray-700" />
                            <TextInput
                                id="name"
                                name="name"
                                value={data.name}
                                className="mt-1 block w-full bg-blue-50 border-blue-200 focus:border-indigo-500 focus:ring-indigo-500 rounded-lg"
                                autoComplete="name"
                                isFocused={true}
                                onChange={(e) => setData('name', e.target.value)}
                                required
                            />
                            <InputError message={errors.name} className="mt-2" />
                        </div>

                        {/* EMAIL */}
                        <div>
                            <InputLabel htmlFor="email" value="Email" className="font-medium text-gray-700" />
                            <TextInput
                                id="email"
                                type="email"
                                name="email"
                                value={data.email}
                                className="mt-1 block w-full bg-blue-50 border-blue-200 focus:border-indigo-500 focus:ring-indigo-500 rounded-lg"
                                autoComplete="username"
                                onChange={(e) => setData('email', e.target.value)}
                                required
                            />
                            <InputError message={errors.email} className="mt-2" />
                        </div>

                        {/* PASSWORD */}
                        <div>
                            <InputLabel htmlFor="password" value="Password" className="font-medium text-gray-700" />
                            <div className="relative">
                                <TextInput
                                    id="password"
                                    type={showPassword ? "text" : "password"}
                                    name="password"
                                    value={data.password}
                                    className="mt-1 block w-full pr-12 bg-blue-50 border-gray-200 focus:border-blue-500 focus:ring-indigo-500 rounded-lg"
                                    autoComplete="new-password"
                                    onChange={(e) => setData('password', e.target.value)}
                                    required
                                />
                                <button
                                    type="button"
                                    onClick={() => setShowPassword(!showPassword)}
                                    className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-700 transition-colors"
                                >
                                    {showPassword ? (
                                        <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                            <path fillRule="evenodd" d="M3.707 2.293a1 1 0 00-1.414 1.414l14 14a1 1 0 001.414-1.414l-1.473-1.473A10.014 10.014 0 0019.542 10C18.268 5.943 14.478 3 10 3a9.958 9.958 0 00-4.512 1.074l-1.78-1.781zm4.261 4.26l1.514 1.515a2.003 2.003 0 012.45 2.45l1.514 1.514a4 4 0 00-5.478-5.478z" clipRule="evenodd" />
                                            <path d="M12.454 16.697L9.75 13.992a4 4 0 01-3.742-3.741L2.335 6.578A9.98 9.98 0 00.458 10c1.274 4.057 5.065 7 9.542 7 .847 0 1.669-.105 2.454-.303z" />
                                        </svg>
                                    ) : (
                                        <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                            <path d="M10 12a2 2 0 100-4 2 2 0 000 4z" />
                                            <path fillRule="evenodd" d="M.458 10C1.732 5.943 5.522 3 10 3s8.268 2.943 9.542 7c-1.274 4.057-5.064 7-9.542 7S1.732 14.057.458 10zM14 10a4 4 0 11-8 0 4 4 0 018 0z" clipRule="evenodd" />
                                        </svg>
                                    )}
                                </button>
                            </div>
                            <InputError message={errors.password} className="mt-2" />
                        </div>

                        {/* CONFIRM PASSWORD */}
                        <div>
                            <InputLabel htmlFor="password_confirmation" value="Confirm Password" className="font-medium text-gray-700" />
                            <div className="relative">
                                <TextInput
                                    id="password_confirmation"
                                    type={showConfirmPassword ? "text" : "password"}
                                    name="password_confirmation"
                                    value={data.password_confirmation}
                                    className="mt-1 block w-full pr-12 bg-blue-50 border-blue-200 focus:border-indigo-500 focus:ring-indigo-500 rounded-lg"
                                    autoComplete="new-password"
                                    onChange={(e) => setData('password_confirmation', e.target.value)}
                                    required
                                />
                                <button
                                    type="button"
                                    onClick={() => setShowConfirmPassword(!showConfirmPassword)}
                                    className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-700 transition-colors"
                                >
                                    {showConfirmPassword ? (
                                        <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                            <path fillRule="evenodd" d="M3.707 2.293a1 1 0 00-1.414 1.414l14 14a1 1 0 001.414-1.414l-1.473-1.473A10.014 10.014 0 0019.542 10C18.268 5.943 14.478 3 10 3a9.958 9.958 0 00-4.512 1.074l-1.78-1.781zm4.261 4.26l1.514 1.515a2.003 2.003 0 012.45 2.45l1.514 1.514a4 4 0 00-5.478-5.478z" clipRule="evenodd" />
                                            <path d="M12.454 16.697L9.75 13.992a4 4 0 01-3.742-3.741L2.335 6.578A9.98 9.98 0 00.458 10c1.274 4.057 5.065 7 9.542 7 .847 0 1.669-.105 2.454-.303z" />
                                        </svg>
                                    ) : (
                                        <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                            <path d="M10 12a2 2 0 100-4 2 2 0 000 4z" />
                                            <path fillRule="evenodd" d="M.458 10C1.732 5.943 5.522 3 10 3s8.268 2.943 9.542 7c-1.274 4.057-5.064 7-9.542 7S1.732 14.057.458 10zM14 10a4 4 0 11-8 0 4 4 0 018 0z" clipRule="evenodd" />
                                        </svg>
                                    )}
                                </button>
                            </div>
                            <InputError message={errors.password_confirmation} className="mt-2" />
                        </div>

                        {/* Hidden fields */}
                        <input type="hidden" name="device_timezone" value={data.device_timezone} />
                        <input type="hidden" name="device_datetime" value={data.device_datetime} />

                        {/* Footer */}
                        <div className="flex flex-col items-center gap-4 mt-8">
                            <Link
                                href={route('login')}
                                className="px-8 py-3 bg-gradient-to-r from-gray-600 to-gray-700 text-white font-bold rounded-xl shadow-lg hover:shadow-gray-500/50 transform hover:scale-105 hover:-translate-y-1 transition-all duration-300"
                            >
                                Already registered? Login
                            </Link>

                            <button
                                type="submit"
                                disabled={processing}
                                className="px-8 py-4 bg-gradient-to-r from-purple-600 via-blue-600 to-indigo-600 text-white font-bold text-lg rounded-xl shadow-2xl hover:shadow-purple-500/50 transform hover:scale-105 hover:-translate-y-1 transition-all duration-300 disabled:opacity-50 disabled:cursor-not-allowed relative overflow-hidden group"
                            >
                                <span className="relative z-10">Register</span>
                                <div className="absolute inset-0 bg-gradient-to-r from-purple-500 via-blue-500 to-indigo-500 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                            </button>
                        </div>
                    </form>
                </motion.div>
            </div>

            {/* Animations for icons */}
            <style>
                {`
                @keyframes float-slow {0%,100%{transform:translateY(0px);}50%{transform:translateY(-20px);}}
                @keyframes float-slower {0%,100%{transform:translateY(0px);}50%{transform:translateY(-10px);}}
                @keyframes float-fast {0%,100%{transform:translateY(0px);}50%{transform:translateY(-30px);}}
                .animate-float-slow {animation: float-slow 5s ease-in-out infinite;}
                .animate-float-slower {animation: float-slower 6s ease-in-out infinite;}
                .animate-float-fast {animation: float-fast 4s ease-in-out infinite;}
                `}
            </style>
        </div>
    );
}
