import { useEffect, useState } from 'react';
import Checkbox from '@/Components/Checkbox';
import InputError from '@/Components/InputError';
import InputLabel from '@/Components/InputLabel';
import PrimaryButton from '@/Components/PrimaryButton';
import TextInput from '@/Components/TextInput';
import { Head, Link, useForm } from '@inertiajs/react';
import { FaBitcoin, FaEthereum, FaDollarSign, FaCoins } from 'react-icons/fa';

export default function Login({ status, canResetPassword }) {
    const [showPassword, setShowPassword] = useState(false);
    const { data, setData, post, processing, errors, reset } = useForm({
        email: '',
        password: '',
        remember: false,
    });

    useEffect(() => {
        return () => reset('password');
    }, []);

    const submit = (e) => {
        e.preventDefault();
        post(route('login'));
    };

    return (
        <div className="min-h-screen min-w-screen flex">
            <Head title="Log in" />

            {/* Bybit / OKX Style Left Panel */}
            <div className="hidden lg:flex flex-1 relative items-center justify-center 
                bg-gradient-to-r from-purple-600 via-blue-600 to-indigo-600 text-white shadow-lg overflow-hidden">

                {/* Faint grid overlay */}
                <div className="absolute inset-0 opacity-[0.07] bg-[url('/images/grid.svg')] bg-cover"></div>

                {/* Soft orange glow blobs */}
                <div className="absolute w-96 h-96 top-[-50px] left-[-80px] 
                    bg-blue-500 rounded-full blur-[140px] opacity-30"></div>

                <div className="absolute w-[500px] h-[500px] bottom-[-100px] right-[-150px] 
                    bg-purple-500 rounded-full blur-[180px] opacity-25"></div>

                {/* Floating crypto icons */}
                <FaBitcoin className="absolute text-yellow-400 text-7xl animate-float-slow 
                    drop-shadow-[0_0_50px_rgba(255,193,7,0.5)] top-32 left-24 opacity-90" />

                <FaEthereum className="absolute text-gray-300 text-6xl animate-float-fast 
                    drop-shadow-[0_0_40px_rgba(200,200,200,0.5)] top-56 left-72 opacity-90" />

                <FaCoins className="absolute text-orange-400 text-7xl animate-float-slower 
                    drop-shadow-[0_0_40px_rgba(249,115,22,0.5)] top-80 left-14 opacity-90" />

                {/* Branding */}
                <div className="relative z-10 text-center px-10">
                    <h1 className="text-white text-6xl font-black tracking-wide">
                        CoinSova
                    </h1>

                    <p className="text-white-400 text-lg mt-5 leading-relaxed max-w-md">
                        Execute trades fast, securely and with deep liquidity —
                        built for professional and high-frequency traders.
                    </p>

                    <div className="mx-auto mt-8 w-32 h-1 bg-gradient-to-r from-orange-400 to-yellow-500 rounded-full"></div>
                </div>
            </div>

            {/* Right login form */}
            <div className="flex flex-1 items-center justify-center bg-white dark:bg-gray-900 w-full h-screen p-10 sm:p-12">
                <div className="w-full max-w-md lg:max-w-lg">
                    <h2 className="text-3xl font-extrabold text-gray-900 dark:text-white mb-6 text-center lg:text-left">
                        Log in to Your Account
                    </h2>

                    {status && (
                        <div className="mb-4 font-medium text-sm text-green-600 animate-pulse text-center lg:text-left">
                            {status}
                        </div>
                    )}

                    <form onSubmit={submit} className="space-y-6">
                        <div>
                            <InputLabel htmlFor="email" value="Email" />
                            <TextInput
                                id="email"
                                type="email"
                                name="email"
                                value={data.email}
                                className="mt-1 block w-full p-3 rounded-xl border border-blue-300 dark:border-indigo-700 focus:ring-2 focus:ring-purple-400 focus:border-indigo-400 dark:bg-blue-800 dark:text-white transition-all duration-300"
                                autoComplete="username"
                                isFocused={true}
                                onChange={(e) => setData('email', e.target.value)}
                            />
                            <InputError message={errors.email} className="mt-2 text-red-500 text-sm" />
                        </div>

                        <div>
                            <InputLabel htmlFor="password" value="Password" />
                            <div className="relative">
                                <TextInput
                                    id="password"
                                    type={showPassword ? "text" : "password"}
                                    name="password"
                                    value={data.password}
                                    className="mt-1 block w-full p-3 pr-12 rounded-xl border border-blue-300 dark:border-indigo-700 focus:ring-2 focus:ring-purple-400 focus:border-blue-400 dark:bg-indigo-800 dark:text-white transition-all duration-300"
                                    autoComplete="current-password"
                                    onChange={(e) => setData('password', e.target.value)}
                                />
                                <button
                                    type="button"
                                    onClick={() => setShowPassword(!showPassword)}
                                    className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 transition-colors"
                                >
                                    {showPassword ? (
                                        <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                            <path fillRule="evenodd" d="M3.707 2.293a1 1 0 00-1.414 1.414l14 14a1 1 0 001.414-1.414l-1.473-1.473A10.014 10.014 0 0019.542 10C18.268 5.943 14.478 3 10 3a9.958 9.958 0 00-4.512 1.074l-1.78-1.781zm4.261 4.26l1.514 1.515a2.003 2.003 0 012.45 2.45l1.514 1.514a4 4 0 00-5.478-5.478z" clipRule="evenodd" />
                                            <path d="M12.454 16.697L9.75 13.992a4 4 0 01-3.742-3.741L2.335 6.578A9.98 9.98 0 00.458 10c1.274 4.057 5.065 7 9.542 7 .847 0 1.669-.105 2.454-.303z" />
                                        </svg>
                                    ) : (
                                        <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                            <path d="M10 12a2 2 0 100-4 2 2 0 000 4z" />
                                            <path fillRule="evenodd" d="M.458 10C1.732 5.943 5.522 3 10 3s8.268 2.943 9.542 7c-1.274 4.057-5.064 7-9.542 7S1.732 14.057.458 10zM14 10a4 4 0 11-8 0 4 4 0 018 0z" clipRule="evenodd" />
                                        </svg>
                                    )}
                                </button>
                            </div>
                            <InputError message={errors.password} className="mt-2 text-red-500 text-sm" />
                        </div>

                        <div className="flex items-center justify-between">
                            <label className="flex items-center space-x-2">
                                <Checkbox
                                    name="remember"
                                    checked={data.remember}
                                    onChange={(e) => setData('remember', e.target.checked)}
                                />
                                <span className="text-sm text-gray-600 dark:text-gray-300">Remember me</span>
                            </label>

                            {canResetPassword && (
                                <Link
                                    href={route('password.request')}
                                    className="text-sm text-blue-500 hover:text-indigo-400 dark:text-indigo-400 dark:hover:text-indigo-300 transition-all duration-300"
                                >
                                    Forgot your password?
                                </Link>
                            )}
                        </div>

                        <button
                            type="submit"
                            disabled={processing}
                            className="w-full py-4 bg-gradient-to-r from-purple-600 via-blue-600 to-indigo-600 text-white font-bold text-lg rounded-xl shadow-2xl hover:shadow-purple-500/50 transform hover:scale-105 hover:-translate-y-1 transition-all duration-300 disabled:opacity-50 disabled:cursor-not-allowed relative overflow-hidden group"
                        >
                            <span className="relative z-10">Log in</span>
                            <div className="absolute inset-0 bg-gradient-to-r from-purple-500 via-blue-500 to-indigo-500 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                        </button>
                    </form>

                    <div className="mt-6 text-center text-sm text-gray-500 dark:text-gray-400">
                        Don’t have an account?{' '}
                        <Link
                            href={route('register')}
                            className="text-indigo-500 hover:text-blue-400 dark:text-blue-400 dark:hover:text-indigo-300 font-medium transition-all duration-300"
                        >
                            Sign up
                        </Link>
                    </div>
                </div>
            </div>

            {/* Animations for icons */}
            <style>
                {`
                @keyframes float-slow {
                    0%, 100% { transform: translateY(0px); }
                    50% { transform: translateY(-20px); }
                }
                @keyframes float-slower {
                    0%, 100% { transform: translateY(0px); }
                    50% { transform: translateY(-10px); }
                }
                @keyframes float-fast {
                    0%, 100% { transform: translateY(0px); }
                    50% { transform: translateY(-30px); }
                }
                .animate-float-slow { animation: float-slow 5s ease-in-out infinite; }
                .animate-float-slower { animation: float-slower 6s ease-in-out infinite; }
                .animate-float-fast { animation: float-fast 4s ease-in-out infinite; }
                `}
            </style>
        </div>
    );
}
