import React, { useState, useEffect, useRef } from 'react';
import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Head, useForm, router } from '@inertiajs/react';
import { createChart, CandlestickSeries } from 'lightweight-charts';
import axios from 'axios';

export default function ChartManipulator({ auth, manipulations }) {
    const [symbol, setSymbol] = useState('BTC');
    const chartContainerRef = useRef(null);
    const chartRef = useRef(null);
    const seriesRef = useRef(null);

    // Form for new manipulation
    const { data, setData, post, processing, errors, reset } = useForm({
        symbol: 'BTC',
        timestamp: Math.floor(Date.now() / 1000) - (Math.floor(Date.now() / 1000) % 60), // Current minute aligned
        open: '',
        high: '',
        low: '',
        close: ''
    });

    // Initialize Chart
    useEffect(() => {
        if (!chartContainerRef.current || typeof window === 'undefined') return;

        let cleanup = null;
        let timeoutId = null;

        // Wait for container to have dimensions
        const initChart = () => {
            if (!chartContainerRef.current) return;

            const container = chartContainerRef.current;
            const width = container.clientWidth || container.offsetWidth || 800;

            // Ensure container has some width before creating chart
            if (width <= 0) {
                // Retry after a short delay
                timeoutId = setTimeout(initChart, 100);
                return;
            }

            try {
                // Verify createChart is available
                if (typeof createChart !== 'function') {
                    console.error('createChart is not a function. Import may have failed.');
                    return;
                }

                const chart = createChart(container, {
                    width: width,
                    height: 400,
                    layout: {
                        background: { color: '#ffffff' },
                        textColor: '#333',
                    },
                    grid: {
                        vertLines: { color: '#f0f0f0' },
                        horzLines: { color: '#f0f0f0' },
                    },
                });

                // Verify chart was created successfully and has the method
                if (!chart) {
                    console.error('Chart creation returned null/undefined');
                    return;
                }

                // lightweight-charts v5 API - use addSeries with CandlestickSeries type
                let candleSeries;
                if (chart.addSeries && typeof chart.addSeries === 'function' && CandlestickSeries) {
                    // v5 API: chart.addSeries(CandlestickSeries, options)
                    candleSeries = chart.addSeries(CandlestickSeries, {
                        upColor: '#26a69a',
                        downColor: '#ef5350',
                        borderVisible: false,
                        wickUpColor: '#26a69a',
                        wickDownColor: '#ef5350',
                    });
                } else if (chart.addCandlestickSeries && typeof chart.addCandlestickSeries === 'function') {
                    // Fallback: v4 API
                    candleSeries = chart.addCandlestickSeries({
                        upColor: '#26a69a',
                        downColor: '#ef5350',
                        borderVisible: false,
                        wickUpColor: '#26a69a',
                        wickDownColor: '#ef5350',
                    });
                } else {
                    console.error('Cannot add candlestick series. Chart object:', chart);
                    console.error('Available methods:', Object.getOwnPropertyNames(Object.getPrototypeOf(chart)));
                    chart.remove();
                    return;
                }

                if (!candleSeries) {
                    console.error('Failed to create candlestick series');
                    chart.remove();
                    return;
                }

                chartRef.current = chart;
                seriesRef.current = candleSeries;

                const handleResize = () => {
                    if (chartContainerRef.current && chartRef.current) {
                        const newWidth = chartContainerRef.current.clientWidth || chartContainerRef.current.offsetWidth;
                        if (newWidth > 0) {
                            chartRef.current.applyOptions({ width: newWidth });
                        }
                    }
                };
                window.addEventListener('resize', handleResize);

                cleanup = () => {
                    window.removeEventListener('resize', handleResize);
                    if (chartRef.current) {
                        chartRef.current.remove();
                        chartRef.current = null;
                    }
                    seriesRef.current = null;
                };
            } catch (error) {
                console.error('Error creating chart:', error);
            }
        };

        // Small delay to ensure DOM is ready
        timeoutId = setTimeout(initChart, 0);

        return () => {
            if (timeoutId) {
                clearTimeout(timeoutId);
            }
            if (cleanup) {
                cleanup();
            }
            if (chartRef.current) {
                chartRef.current.remove();
                chartRef.current = null;
            }
            seriesRef.current = null;
        };
    }, []);

    // Fetch Data on symbol change
    useEffect(() => {
        if (!seriesRef.current) return;

        const fetchChartData = async () => {
            try {
                const res = await axios.get('/api/chart-data', { params: { symbol } });
                if (seriesRef.current && res.data && Array.isArray(res.data)) {
                    seriesRef.current.setData(res.data);
                }
            } catch (err) {
                if (err.response?.status === 401) {
                    window.location.href = "/login";
                    return;
                }
                console.error('Error fetching chart data:', err);
            }
        };

        fetchChartData();
        const interval = setInterval(fetchChartData, 5000); // Poll for updates
        return () => clearInterval(interval);
    }, [symbol]);

    const submit = (e) => {
        e.preventDefault();
        post(route('admin.charts.store'), {
            onSuccess: () => {
                reset('open', 'high', 'low', 'close');
                // Refresh chart manually to show new candle immediately
            }
        });
    };

    const setCurrentTime = () => {
        const now = Math.floor(Date.now() / 1000);
        const aligned = now - (now % 60);
        setData('timestamp', aligned);
    };

    return (
        <AuthenticatedLayout user={auth.user}>
            <Head title="Chart Manipulation" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8 space-y-6">

                    {/* 1. Controller Header */}
                    <div className="flex justify-between items-center bg-white p-6 rounded-lg shadow">
                        <h2 className="text-2xl font-bold text-gray-800">Chart Master Control</h2>
                        <select
                            value={symbol}
                            onChange={e => { setSymbol(e.target.value); setData('symbol', e.target.value); }}
                            className="p-2 border rounded-md"
                        >
                            {['BTC', 'ETH', 'SOL', 'BNB', 'XRP'].map(s => (
                                <option key={s} value={s}>{s}</option>
                            ))}
                        </select>
                    </div>

                    <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">

                        {/* 2. Chart Preview */}
                        <div className="lg:col-span-2 bg-white p-4 rounded-lg shadow">
                            <h3 className="font-bold mb-4">Live Preview ({symbol}USDT)</h3>
                            <div ref={chartContainerRef} className="w-full h-[400px]" />
                        </div>

                        {/* 3. Manipulation Form */}
                        <div className="bg-white p-6 rounded-lg shadow space-y-4">
                            <h3 className="font-bold text-lg mb-4">Manipulate Candle</h3>
                            <form onSubmit={submit} className="space-y-4">
                                <div>
                                    <label className="block text-sm font-medium text-gray-700">Timestamp (Unix)</label>
                                    <div className="flex gap-2">
                                        <input
                                            type="number"
                                            value={data.timestamp}
                                            onChange={e => setData('timestamp', e.target.value)}
                                            className="w-full border-gray-300 rounded-md shadow-sm"
                                        />
                                        <button
                                            type="button"
                                            onClick={setCurrentTime}
                                            className="px-2 py-1 bg-gray-200 text-xs rounded hover:bg-gray-300"
                                        >
                                            Now
                                        </button>
                                    </div>
                                    <p className="text-xs text-gray-500 mt-1">
                                        {new Date(data.timestamp * 1000).toLocaleString()}
                                    </p>
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">Open</label>
                                        <input
                                            type="number" step="0.0001"
                                            value={data.open}
                                            onChange={e => setData('open', e.target.value)}
                                            className="w-full border-gray-300 rounded-md shadow-sm"
                                        />
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">Close</label>
                                        <input
                                            type="number" step="0.0001"
                                            value={data.close}
                                            onChange={e => setData('close', e.target.value)}
                                            className="w-full border-gray-300 rounded-md shadow-sm"
                                        />
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">High</label>
                                        <input
                                            type="number" step="0.0001"
                                            value={data.high}
                                            onChange={e => setData('high', e.target.value)}
                                            className="w-full border-gray-300 rounded-md shadow-sm"
                                        />
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">Low</label>
                                        <input
                                            type="number" step="0.0001"
                                            value={data.low}
                                            onChange={e => setData('low', e.target.value)}
                                            className="w-full border-gray-300 rounded-md shadow-sm"
                                        />
                                    </div>
                                </div>

                                <button
                                    type="submit"
                                    disabled={processing}
                                    className="w-full py-2 bg-red-600 hover:bg-red-700 text-white font-bold rounded shadow transition-colors"
                                >
                                    Overrule Market Data
                                </button>
                            </form>
                        </div>
                    </div>

                    {/* 4. Active Manipulations Table */}
                    <div className="bg-white p-6 rounded-lg shadow">
                        <h3 className="font-bold mb-4">Active Manipulations</h3>
                        <table className="w-full text-sm text-left">
                            <thead>
                                <tr className="border-b">
                                    <th className="py-2">Symbol</th>
                                    <th className="py-2">Time</th>
                                    <th className="py-2">Open</th>
                                    <th className="py-2">High</th>
                                    <th className="py-2">Low</th>
                                    <th className="py-2">Close</th>
                                    <th className="py-2">Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                {manipulations.map(m => (
                                    <tr key={m.id} className="border-b hover:bg-gray-50">
                                        <td className="py-2 font-bold">{m.symbol}</td>
                                        <td className="py-2">{new Date(m.timestamp * 1000).toLocaleString()}</td>
                                        <td className="py-2">{Number(m.open).toFixed(2)}</td>
                                        <td className="py-2">{Number(m.high).toFixed(2)}</td>
                                        <td className="py-2">{Number(m.low).toFixed(2)}</td>
                                        <td className="py-2">{Number(m.close).toFixed(2)}</td>
                                        <td className="py-2">
                                            <span className="px-2 py-1 bg-green-100 text-green-800 rounded-full text-xs">Active</span>
                                        </td>
                                    </tr>
                                ))}
                                {manipulations.length === 0 && (
                                    <tr>
                                        <td colSpan="7" className="py-4 text-center text-gray-500">No active manipulations.</td>
                                    </tr>
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
}
