import { useState, useEffect } from "react";
import { router } from "@inertiajs/react";
import Navbar from "@/Components/Navbar";
import { FiSearch, FiPlus, FiEdit, FiTrash, FiX, FiCheck, FiCheckCircle, FiXCircle } from "react-icons/fi";

export default function AdminWallet({ auth, transactions, users }) {
    const [search, setSearch] = useState("");
    const [previewImage, setPreviewImage] = useState(null);
    const [showModal, setShowModal] = useState(false);
    const [modalMode, setModalMode] = useState("create");
    const [editingTx, setEditingTx] = useState(null);
    const [form, setForm] = useState({ user_id: "", crypto: "BTC", amount: "", type: "deposit", status: "pending" });
    const [theme, setTheme] = useState(localStorage.getItem("theme") || "light");

    const [notification, setNotification] = useState({ message: "", type: "" });

    const showNotification = (message, type = "success") => {
        setNotification({ message, type });
        setTimeout(() => setNotification({ message: "", type: "" }), 3000);
    };

    const handleAction = (id, action) => {
        router.post(`/admin-wallet/${id}/${action}`, {}, {
            preserveScroll: true,
            onSuccess: () => showNotification(`Transaction #${id} ${action}d successfully!`)
        });
    };

    // Theme Synchronization
    useEffect(() => {
        const handleThemeChange = (e) => setTheme(e.detail);
        window.addEventListener('themeChanged', handleThemeChange);

        document.documentElement.classList.toggle("dark", theme === "dark");
        localStorage.setItem("theme", theme);

        return () => window.removeEventListener('themeChanged', handleThemeChange);
    }, [theme]);

    // Polling for real-time updates
    useEffect(() => {
        const interval = setInterval(() => {
            router.reload({ only: ['transactions'], preserveScroll: true });
        }, 3000); // Poll every 3 seconds

        return () => clearInterval(interval);
    }, []);

    const openCreate = () => {
        setModalMode("create");
        setForm({ user_id: users.length > 0 ? users[0].id : "", crypto: "BTC", amount: "", type: "deposit", status: "pending" });
        setShowModal(true);
    };

    const openEdit = (tx) => {
        setModalMode("edit");
        setEditingTx(tx);
        setForm({ user_id: tx.user_id, crypto: tx.crypto, amount: tx.amount, type: tx.type, status: tx.status });
        setShowModal(true);
    };

    const handleDelete = (id) => {
        if (confirm("Are you sure you want to delete this transaction?")) {
            router.delete(`/admin-wallet/${id}`, {
                preserveScroll: true,
                onSuccess: () => showNotification("Transaction deleted successfully!")
            });
        }
    };

    const submitForm = (e) => {
        e.preventDefault();
        if (modalMode === "create") {
            router.post('/admin-wallet', form, {
                onSuccess: () => {
                    setShowModal(false);
                    showNotification("Transaction created successfully!");
                },
                preserveScroll: true
            });
        } else {
            router.put(`/admin-wallet/${editingTx.id}`, form, {
                onSuccess: () => {
                    setShowModal(false);
                    showNotification("Transaction updated successfully!");
                },
                preserveScroll: true
            });
        }
    };

    // Filter + Sort
    const filteredTransactions = transactions.filter(
        (tx) =>
            tx.user.name.toLowerCase().includes(search.toLowerCase()) ||
            tx.crypto.toLowerCase().includes(search.toLowerCase()) ||
            String(tx.amount).includes(search)
    );
    const sortedTransactions = filteredTransactions.sort((a, b) => b.id - a.id); // Show newest first

    const statusColors = {
        pending: "bg-yellow-400 text-yellow-900",
        approved: "bg-green-400 text-green-900",
        declined: "bg-red-400 text-red-900",
    };

    return (
        <div className={`min-h-screen transition-colors duration-500 ${theme === "dark"
            ? "bg-gray-900 text-gray-100"
            : "bg-gradient-to-r from-purple-100 via-pink-50 to-white text-gray-900"
            }`}>
            {/* Navbar */}
            <Navbar user={auth.user} />

            {/* Notification Toast */}
            {notification.message && (
                <div className={`fixed top-20 left-1/2 -translate-x-1/2 px-6 py-4 rounded-xl shadow-2xl z-50 text-white font-bold animate-slide-in ${notification.type === 'success' ? 'bg-green-500' : 'bg-red-500'
                    }`}>
                    {notification.message}
                </div>
            )}

            {/* MAIN CONTENT */}
            <main className="p-4 md:p-6 w-full max-w-7xl mx-auto">
                <div className="mb-4 flex flex-col md:flex-row justify-between items-center gap-4">
                    <div className="relative w-full md:w-1/3">
                        <FiSearch className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400" />
                        <input
                            type="text"
                            placeholder="Search user or crypto..."
                            value={search}
                            onChange={(e) => setSearch(e.target.value)}
                            className="w-full pl-10 pr-4 py-2 rounded-xl border shadow-md focus:outline-none focus:ring-2 bg-white border-gray-300 text-gray-900 focus:ring-purple-400 dark:bg-gray-800 dark:border-gray-700 dark:text-gray-200"
                        />
                    </div>
                    <button
                        onClick={openCreate}
                        className="flex items-center gap-2 bg-gradient-to-r from-purple-600 to-indigo-600 text-white px-4 py-2 rounded-lg shadow hover:opacity-90 transition"
                    >
                        <FiPlus /> Add Transaction
                    </button>
                </div>

                {/* Desktop Table View */}
                <div className="hidden md:block overflow-x-auto">
                    <table className="w-full border-collapse rounded-lg shadow-lg bg-white dark:bg-gray-800">
                        <thead>
                            <tr className="bg-gray-100 dark:bg-gray-700">
                                <th className="px-4 py-3 text-left font-semibold">ID</th>
                                <th className="px-4 py-3 text-left font-semibold">User</th>
                                <th className="px-4 py-3 text-left font-semibold">Crypto</th>
                                <th className="px-4 py-3 text-left font-semibold">Amount</th>
                                <th className="px-4 py-3 text-left font-semibold">Type</th>
                                <th className="px-4 py-3 text-left font-semibold">Wallet Address</th>
                                <th className="px-4 py-3 text-left font-semibold">Date</th>
                                <th className="px-4 py-3 text-left font-semibold">Status</th>
                                <th className="px-4 py-3 text-left font-semibold">Proof</th>
                                <th className="px-4 py-3 text-left font-semibold text-center">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            {sortedTransactions.length === 0 && (
                                <tr>
                                    <td colSpan="10" className="px-4 py-8 text-center text-gray-500 italic">
                                        No transactions found
                                    </td>
                                </tr>
                            )}
                            {sortedTransactions.map((tx) => (
                                <tr key={tx.id} className="border-b border-gray-200 hover:bg-gray-50 transition dark:border-gray-700 dark:hover:bg-gray-700">
                                    <td className="px-4 py-3 text-purple-400 font-mono">#{tx.id}</td>
                                    <td className="px-4 py-3 font-medium">{tx.user.name}</td>
                                    <td className="px-4 py-3 uppercase">{tx.crypto}</td>
                                    <td className="px-4 py-3 font-mono">${tx.amount}</td>
                                    <td className="px-4 py-3 capitalize">{tx.type}</td>
                                    <td className="px-4 py-3 text-xs font-mono text-gray-500 max-w-[150px] truncate" title={tx.wallet_address}>{tx.wallet_address || '-'}</td>
                                    <td className="px-4 py-3 text-sm">{new Date(tx.created_at).toLocaleDateString()}</td>
                                    <td className="px-4 py-3 text-center">
                                        <span className={`px-2 py-1 rounded-full text-[10px] font-bold uppercase tracking-wider ${statusColors[tx.status]}`}>
                                            {tx.status}
                                        </span>
                                    </td>
                                    <td className="px-4 py-3 text-center">
                                        {tx.proof ? (
                                            <img
                                                src={tx.proof_url || `/storage/${tx.proof}`}
                                                alt="Proof"
                                                className="w-12 h-12 object-cover rounded shadow-sm mx-auto cursor-pointer hover:scale-110 transition"
                                                onClick={() => setPreviewImage(tx.proof_url || `/storage/${tx.proof}`)}
                                            />
                                        ) : (
                                            <span className="text-gray-500 italic text-sm">None</span>
                                        )}
                                    </td>
                                    <td className="px-4 py-3">
                                        <div className="flex justify-center gap-2">
                                            <button
                                                onClick={() => handleAction(tx.id, "approve")}
                                                className="p-2 rounded-lg bg-green-600 hover:bg-green-500 text-white transition shadow-md hover:scale-110"
                                                title="Approve"
                                            >
                                                <FiCheckCircle size={18} />
                                            </button>
                                            <button
                                                onClick={() => handleAction(tx.id, "decline")}
                                                className="p-2 rounded-lg bg-red-600 hover:bg-red-500 text-white transition shadow-md hover:scale-110"
                                                title="Decline"
                                            >
                                                <FiXCircle size={18} />
                                            </button>
                                            <button
                                                onClick={() => openEdit(tx)}
                                                className="p-2 bg-blue-600 text-white rounded-lg hover:bg-blue-500 transition shadow-md hover:scale-110"
                                                title="Edit"
                                            >
                                                <FiEdit />
                                            </button>
                                            <button
                                                onClick={() => handleDelete(tx.id)}
                                                className="p-2 bg-gray-500 text-white rounded-lg hover:bg-red-600 transition shadow-md hover:scale-110"
                                                title="Delete"
                                            >
                                                <FiTrash />
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>

                {/* Mobile Cards View */}
                <div className="md:hidden grid grid-cols-1 gap-6">
                    {sortedTransactions.length === 0 && (
                        <p className="text-center text-gray-500 italic">
                            No transactions found
                        </p>
                    )}

                    {sortedTransactions.map((tx) => {
                        const cardGradient =
                            tx.status === "pending"
                                ? "from-yellow-100 to-yellow-200"
                                : tx.status === "approved"
                                    ? "from-green-100 to-green-200"
                                    : "from-red-100 to-red-200";

                        return (
                            <div
                                key={tx.id}
                                className={`relative p-6 rounded-2xl shadow-lg border backdrop-blur-md transition transform hover:scale-[1.02] hover:shadow-2xl bg-gradient-to-br border-gray-200 dark:bg-none dark:bg-gray-800 dark:border-gray-700 ${cardGradient}`}
                            >
                                {/* Header */}
                                <div className="flex justify-between items-center mb-3">
                                    <span className="font-bold text-lg dark:text-gray-100">{tx.user.name}</span>
                                    <span
                                        className={`px-3 py-1 rounded-full text-sm font-semibold ${statusColors[tx.status]}`}
                                    >
                                        {tx.status}
                                    </span>
                                </div>

                                {/* Details */}
                                <div className="dark:text-gray-300">
                                    <p><strong>Crypto:</strong> {tx.crypto}</p>
                                    <p><strong>Amount:</strong> ${tx.amount}</p>
                                    <p><strong>Type:</strong> {tx.type}</p>
                                    {tx.wallet_address && <p className="truncate text-xs"><strong className="text-base">Address:</strong> {tx.wallet_address}</p>}
                                    <p className="mb-3"><strong>Date:</strong> {tx.created_at}</p>
                                </div>

                                {/* Proof */}
                                <div className="mt-3">
                                    <strong className="dark:text-gray-300">Proof:</strong>
                                    {!tx.proof && (
                                        <div className="w-full h-40 flex items-center justify-center bg-gray-200 rounded-lg text-gray-500 italic mt-2 dark:bg-gray-700 dark:text-gray-400">
                                            No proof uploaded
                                        </div>
                                    )}
                                    {tx.proof && (
                                        <>
                                            {tx.proof_url ? (
                                                <>
                                                    {tx.proof.endsWith(".pdf") || tx.proof_url.endsWith(".pdf") ? (
                                                        <a
                                                            href={tx.proof_url}
                                                            target="_blank"
                                                            rel="noopener noreferrer"
                                                            className="px-4 py-2 inline-block mt-2 bg-blue-600 text-white rounded-lg shadow hover:bg-blue-700 transition"
                                                        >
                                                            View PDF
                                                        </a>
                                                    ) : (
                                                        <img
                                                            src={tx.proof_url}
                                                            alt="Proof of transaction"
                                                            className="w-full h-40 object-cover rounded-lg mt-2 cursor-pointer shadow hover:opacity-90 transition"
                                                            onClick={() => setPreviewImage(tx.proof_url)}
                                                        />
                                                    )}
                                                </>
                                            ) : (
                                                <img
                                                    src={`/storage/${tx.proof}`}
                                                    alt="Proof of transaction"
                                                    className="w-full h-40 object-cover rounded-lg mt-2 cursor-pointer shadow hover:opacity-90 transition"
                                                    onClick={() => setPreviewImage(`/storage/${tx.proof}`)}
                                                />
                                            )}
                                        </>
                                    )}
                                </div>

                                {/* Actions */}
                                <div className="flex gap-2 mt-4">
                                    <button
                                        onClick={() => handleAction(tx.id, "approve")}
                                        className="flex-1 p-2 rounded-xl shadow-md text-white font-semibold transition bg-green-600 hover:bg-green-500"
                                        title="Approve"
                                    >
                                        <FiCheckCircle className="mx-auto" size={20} />
                                    </button>
                                    <button
                                        onClick={() => handleAction(tx.id, "decline")}
                                        className="flex-1 p-2 rounded-xl shadow-md text-white font-semibold transition bg-red-600 hover:bg-red-500"
                                        title="Decline"
                                    >
                                        <FiXCircle className="mx-auto" size={20} />
                                    </button>
                                    <button onClick={() => openEdit(tx)} className="p-2 bg-blue-600 text-white rounded-xl shadow-md hover:bg-blue-500 flex-1 flex justify-center items-center transition"><FiEdit size={20} /></button>
                                    <button onClick={() => handleDelete(tx.id)} className="p-2 bg-gray-500 text-white rounded-xl shadow-md hover:bg-red-600 flex-1 flex justify-center items-center transition"><FiTrash size={20} /></button>
                                </div>
                            </div>
                        );
                    })}
                </div>
            </main>

            {/* Image Preview Modal */}
            {previewImage && (
                <div
                    className="fixed inset-0 bg-black/70 flex items-center justify-center z-50"
                    onClick={() => setPreviewImage(null)}
                >
                    <img
                        src={previewImage}
                        className="max-w-3xl max-h-[90vh] rounded-lg shadow-lg"
                    />
                </div>
            )}

            {/* Create/Edit Modal */}
            {showModal && (
                <div className="fixed inset-0 bg-black/70 flex items-center justify-center z-50">
                    <div className="w-full max-w-lg p-6 rounded-2xl shadow-2xl bg-white text-gray-900 dark:bg-gray-800 dark:text-white">
                        <div className="flex justify-between items-center mb-6">
                            <h3 className="text-xl font-bold">{modalMode === 'create' ? 'Add Transaction' : 'Edit Transaction'}</h3>
                            <button onClick={() => setShowModal(false)} className="text-gray-500 hover:text-red-500"><FiX size={24} /></button>
                        </div>
                        <form onSubmit={submitForm} className="space-y-4">
                            {modalMode === 'create' && (
                                <div>
                                    <label className="block text-sm font-bold mb-1">User</label>
                                    <select
                                        value={form.user_id}
                                        onChange={e => setForm({ ...form, user_id: e.target.value })}
                                        className="w-full p-3 rounded-lg bg-gray-50 dark:bg-gray-700 border border-gray-200 dark:border-gray-600"
                                        required
                                    >
                                        <option value="">Select User</option>
                                        {users && users.map(u => (
                                            <option key={u.id} value={u.id}>{u.name} ({u.email})</option>
                                        ))}
                                    </select>
                                </div>
                            )}

                            <div className="grid grid-cols-2 gap-4">
                                <div>
                                    <label className="block text-sm font-bold mb-1">Crypto</label>
                                    <select
                                        value={form.crypto}
                                        onChange={e => setForm({ ...form, crypto: e.target.value })}
                                        className="w-full p-3 rounded-lg bg-gray-50 dark:bg-gray-700 border border-gray-200 dark:border-gray-600"
                                    >
                                        {['BTC', 'ETH', 'SOL', 'XRP', 'USDT', 'USDC', 'DOGE'].map(c => <option key={c} value={c}>{c}</option>)}
                                    </select>
                                </div>
                                <div>
                                    <label className="block text-sm font-bold mb-1">Amount</label>
                                    <input
                                        type="number"
                                        step="any"
                                        value={form.amount}
                                        onChange={e => setForm({ ...form, amount: e.target.value })}
                                        className="w-full p-3 rounded-lg bg-gray-50 dark:bg-gray-700 border border-gray-200 dark:border-gray-600"
                                        required
                                    />
                                </div>
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <div>
                                    <label className="block text-sm font-bold mb-1">Type</label>
                                    <select
                                        value={form.type}
                                        onChange={e => setForm({ ...form, type: e.target.value })}
                                        className="w-full p-3 rounded-lg bg-gray-50 dark:bg-gray-700 border border-gray-200 dark:border-gray-600"
                                        disabled={modalMode === 'edit'}
                                    >
                                        <option value="deposit">Deposit</option>
                                        <option value="withdraw">Withdraw</option>
                                    </select>
                                </div>
                                <div>
                                    <label className="block text-sm font-bold mb-1">Status</label>
                                    <select
                                        value={form.status}
                                        onChange={e => setForm({ ...form, status: e.target.value })}
                                        className="w-full p-3 rounded-lg bg-gray-50 dark:bg-gray-700 border border-gray-200 dark:border-gray-600"
                                    >
                                        <option value="pending">Pending</option>
                                        <option value="approved">Approved</option>
                                        <option value="declined">Declined</option>
                                    </select>
                                </div>
                            </div>

                            <button type="submit" className="w-full bg-blue-600 hover:bg-blue-700 text-white font-bold py-3 rounded-xl transition shadow-lg mt-4">
                                {modalMode === 'create' ? 'Create Transaction' : 'Update Transaction'}
                            </button>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
