import React, { useState, useEffect } from "react";
import axios from "axios";
import Navbar from "@/Components/Navbar";
import { usePage, router } from "@inertiajs/react";
import { FiCheckCircle, FiXCircle, FiEdit, FiTrash2, FiSearch, FiPlus, FiX } from "react-icons/fi";

export default function AdminTradeDashboard({ trades: initialTrades, users: initialUsers = [] }) {
    const { auth } = usePage().props; // Get auth user for Navbar
    const [trades, setTrades] = useState(initialTrades || []);
    const [users] = useState(initialUsers || []);

    // Sync state with reloaded props (useful for polling/external updates)
    useEffect(() => {
        if (initialTrades) {
            setTrades(initialTrades);
        }
    }, [initialTrades]);

    // Polling for real-time updates
    useEffect(() => {
        const interval = setInterval(() => {
            router.reload({ only: ['trades', 'users'], preserveScroll: true });
        }, 3000); // Poll every 3 seconds

        return () => clearInterval(interval);
    }, []);

    const [search, setSearch] = useState("");
    const [theme, setTheme] = useState(localStorage.getItem("theme") || "light");
    const [previewImage, setPreviewImage] = useState(null);

    // Modal states
    const [showAddModal, setShowAddModal] = useState(false);
    const [showEditModal, setShowEditModal] = useState(false);
    const [showDeleteModal, setShowDeleteModal] = useState(false);
    const [selectedTrade, setSelectedTrade] = useState(null);

    // Form state
    const [formData, setFormData] = useState({
        user_id: "",
        asset: "",
        amount: "",
        direction: "buy",
        leverage: "",
        status: "pending",
        profit: "",
        purchase_price: "",
        delivery_price: "",
    });

    // Theme Synchronization
    useEffect(() => {
        const handleThemeChange = (e) => setTheme(e.detail);
        window.addEventListener('themeChanged', handleThemeChange);

        document.documentElement.classList.toggle("dark", theme === "dark");
        localStorage.setItem("theme", theme);

        return () => window.removeEventListener('themeChanged', handleThemeChange);
    }, [theme]);

    const filteredTrades = trades.filter((trade) =>
        trade.user?.name?.toLowerCase().includes(search.toLowerCase()) ||
        trade.asset?.toLowerCase().includes(search.toLowerCase()) ||
        String(trade.amount).includes(search) ||
        String(trade.id).includes(search)
    );
    const sortedTrades = [...filteredTrades].sort((a, b) => b.id - a.id);

    const [notification, setNotification] = useState({ message: "", type: "" });

    const showNotification = (message, type = "success") => {
        setNotification({ message, type });
        setTimeout(() => setNotification({ message: "", type: "" }), 3000);
    };

    const handleTradeAction = async (id, action) => {
        try {
            await axios.post(`/admin-trade/${id}/${action}`);
            setTrades((prev) =>
                prev.map((t) =>
                    t.id === id
                        ? { ...t, status: action === "approve" ? "approved" : "declined" }
                        : t
                )
            );
            showNotification(`Trade #${id} ${action === "approve" ? "approved" : "declined"} successfully!`);
        } catch (error) {
            console.error(error);
            showNotification("Something went wrong while updating the trade.", "error");
        }
    };

    const handleAdd = () => {
        setFormData({
            user_id: "",
            asset: "",
            amount: "",
            direction: "buy",
            leverage: "",
            status: "pending",
            profit: "",
            purchase_price: "",
            delivery_price: "",
        });
        setShowAddModal(true);
    };

    const handleEdit = (trade) => {
        setSelectedTrade(trade);
        setFormData({
            user_id: trade.user_id || "",
            asset: trade.asset || "",
            amount: trade.amount || "",
            direction: trade.direction || "buy",
            leverage: trade.leverage || "",
            status: trade.status || "pending",
            profit: trade.profit || "",
            purchase_price: trade.purchase_price || "",
            delivery_price: trade.delivery_price || "",
        });
        setShowEditModal(true);
    };

    const handleDelete = (trade) => {
        setSelectedTrade(trade);
        setShowDeleteModal(true);
    };

    const handleSubmitAdd = async (e) => {
        e.preventDefault();
        try {
            const response = await axios.post("/admin-trade", formData);
            setTrades((prev) => [response.data.trade, ...prev]);
            setShowAddModal(false);
            showNotification("Trade added successfully!");
        } catch (error) {
            console.error(error);
            showNotification("Something went wrong while adding the trade.", "error");
        }
    };

    const handleSubmitEdit = async (e) => {
        e.preventDefault();
        try {
            // Ensure numeric values are sent correctly
            const submitData = {
                ...formData,
                profit: formData.profit === "" ? null : Number(formData.profit),
                amount: Number(formData.amount),
                leverage: formData.leverage === "" ? null : Number(formData.leverage),
                purchase_price: formData.purchase_price === "" ? null : Number(formData.purchase_price),
                delivery_price: formData.delivery_price === "" ? null : Number(formData.delivery_price),
            };

            const response = await axios.put(`/admin-trade/${selectedTrade.id}`, submitData);
            setTrades((prev) =>
                prev.map((t) => (t.id === selectedTrade.id ? response.data.trade : t))
            );
            setShowEditModal(false);
            setSelectedTrade(null);
            showNotification("Trade updated successfully!");
        } catch (error) {
            console.error(error);
            showNotification("Something went wrong while updating the trade.", "error");
        }
    };

    const handleConfirmDelete = async () => {
        try {
            await axios.delete(`/admin-trade/${selectedTrade.id}`);
            setTrades((prev) => prev.filter((t) => t.id !== selectedTrade.id));
            setShowDeleteModal(false);
            setSelectedTrade(null);
            showNotification("Trade deleted successfully!");
        } catch (error) {
            console.error(error);
            showNotification("Something went wrong while deleting the trade.", "error");
        }
    };

    const statusColors = {
        pending: "bg-yellow-700 text-yellow-200",
        approved: "bg-green-700 text-green-200",
        declined: "bg-red-700 text-red-200",
    };

    return (
        <div className={`min-h-screen transition-colors duration-500 ${theme === "dark"
            ? "bg-gray-900 text-gray-100"
            : "bg-gray-100 text-gray-900"
            }`}>
            {/* Navbar */}
            <Navbar user={auth.user} />

            {/* Notification Toast */}
            {notification.message && (
                <div className={`fixed top-20 left-1/2 -translate-x-1/2 px-6 py-4 rounded-xl shadow-2xl z-50 text-white font-bold animate-slide-in ${notification.type === 'success' ? 'bg-green-500' : 'bg-red-500'
                    }`}>
                    {notification.message}
                </div>
            )}

            {/* Main Content */}
            <main className="p-4 md:p-6 w-full max-w-7xl mx-auto">
                {/* Header with Add Button */}
                <div className="mb-4 flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                    <div className="relative w-full md:w-1/3">
                        <FiSearch className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400" />
                        <input
                            type="text"
                            placeholder="Search user, asset, amount or ID..."
                            value={search}
                            onChange={(e) => setSearch(e.target.value)}
                            className={`w-full pl-10 pr-4 py-2 rounded-xl border shadow-md focus:outline-none focus:ring-2 transition ${theme === "dark"
                                ? "bg-gray-800 border-gray-700 text-gray-200 focus:ring-purple-400"
                                : "bg-white border-gray-300 text-gray-900 focus:ring-purple-400"
                                }`}
                        />
                    </div>
                    <button
                        onClick={handleAdd}
                        className="flex items-center gap-2 px-6 py-2 rounded-xl font-semibold bg-purple-600 hover:bg-purple-500 text-white transition transform hover:scale-105 shadow-md"
                    >
                        <FiPlus /> Add Trade
                    </button>
                </div>

                {/* Desktop Table View */}
                <div className="hidden md:block overflow-x-auto">
                    <table className={`w-full border-collapse ${theme === "dark" ? "bg-gray-800" : "bg-white"} rounded-lg shadow-lg`}>
                        <thead>
                            <tr className={`${theme === "dark" ? "bg-gray-700 font-bold" : "bg-gray-100 font-bold"}`}>
                                <th className="px-4 py-3 text-left">ID</th>
                                <th className="px-4 py-3 text-left">User</th>
                                <th className="px-4 py-3 text-left">Asset</th>
                                <th className="px-4 py-3 text-left">Amount</th>
                                <th className="px-4 py-3 text-left">Direction</th>
                                <th className="px-4 py-3 text-left">Leverage</th>
                                <th className="px-4 py-3 text-left">Profit</th>
                                <th className="px-4 py-3 text-left">Status</th>
                                <th className="px-4 py-3 text-left text-center">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            {sortedTrades.length === 0 && (
                                <tr>
                                    <td colSpan="8" className="px-4 py-8 text-center text-gray-500 italic">
                                        No trades found
                                    </td>
                                </tr>
                            )}
                            {sortedTrades.map((trade) => (
                                <tr key={trade.id} className={`border-b ${theme === "dark" ? "border-gray-700 hover:bg-gray-700" : "border-gray-200 hover:bg-gray-50"} transition`}>
                                    <td className="px-4 py-3 text-purple-400 font-mono">#{trade.id}</td>
                                    <td className="px-4 py-3 font-medium">{trade.user?.name || "Unknown"}</td>
                                    <td className="px-4 py-3 uppercase">{trade.asset}</td>
                                    <td className="px-4 py-3 font-mono">${trade.amount}</td>
                                    <td className="px-4 py-3">
                                        <span className={`font-bold ${trade.direction === 'buy' ? 'text-green-500' : 'text-red-500'}`}>
                                            {trade.direction?.toUpperCase()}
                                        </span>
                                    </td>
                                    <td className="px-4 py-3 opacity-70">1:{trade.leverage}</td>
                                    <td className={`px-4 py-3 font-mono ${Number(trade.profit) >= 0 ? "text-green-500" : "text-red-500"}`}>
                                        {trade.profit ? `${Number(trade.profit).toFixed(2)}` : "0.00"}
                                    </td>
                                    <td className="px-4 py-3">
                                        <span className={`px-2 py-1 rounded-full text-[10px] font-bold uppercase tracking-wider ${statusColors[trade.status]}`}>
                                            {trade.status}
                                        </span>
                                    </td>
                                    <td className="px-4 py-3">
                                        <div className="flex justify-center gap-2">
                                            <button
                                                onClick={async () => {
                                                    await handleTradeAction(trade.id, "approve");
                                                }}
                                                className="p-2 rounded-lg bg-green-600 hover:bg-green-500 text-white transition shadow-md hover:scale-110"
                                                title="Approve"
                                            >
                                                <FiCheckCircle size={18} />
                                            </button>
                                            <button
                                                onClick={async () => {
                                                    await handleTradeAction(trade.id, "decline");
                                                }}
                                                className="p-2 rounded-lg bg-red-600 hover:bg-red-500 text-white transition shadow-md hover:scale-110"
                                                title="Decline"
                                            >
                                                <FiXCircle size={18} />
                                            </button>
                                            <button
                                                onClick={() => handleEdit(trade)}
                                                className="p-2 rounded-lg bg-blue-600 hover:bg-blue-500 text-white transition shadow-md hover:scale-110"
                                                title="Edit"
                                            >
                                                <FiEdit size={18} />
                                            </button>
                                            <button
                                                onClick={() => handleDelete(trade)}
                                                className="p-2 rounded-lg bg-gray-500 hover:bg-red-600 text-white transition shadow-md hover:scale-110"
                                                title="Delete"
                                            >
                                                <FiTrash2 size={18} />
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>

                {/* Mobile Cards View */}
                <div className="md:hidden grid grid-cols-1 gap-6">
                    {sortedTrades.length === 0 && (
                        <p className="text-center text-gray-500 italic">
                            No trades found.
                        </p>
                    )}

                    {sortedTrades.map((trade) => {
                        const cardGradient = trade.status === "pending"
                            ? "from-yellow-800 to-yellow-900"
                            : trade.status === "approved"
                                ? "from-green-800 to-green-900"
                                : "from-red-800 to-red-900";

                        return (
                            <div
                                key={trade.id}
                                className={`relative p-6 rounded-2xl shadow-lg border backdrop-blur-md transition transform hover:scale-105 hover:shadow-2xl
                                    ${theme === "dark"
                                        ? "bg-gray-800 border-gray-700 text-gray-100"
                                        : `bg-gradient-to-br ${cardGradient} border-gray-700 text-gray-100`
                                    }`}
                            >
                                <div className="flex justify-between items-center mb-4">
                                    <span className="font-bold text-lg">ID: {trade.id}</span>
                                    <span className={`px-3 py-1 rounded-full text-sm font-semibold ${statusColors[trade.status]}`}>
                                        {trade.status.charAt(0).toUpperCase() + trade.status.slice(1)}
                                    </span>
                                </div>

                                <p className="mb-1"><span className="font-semibold">User:</span> {trade.user?.name || "Unknown"}</p>
                                <p className="mb-1"><span className="font-semibold">Asset:</span> {trade.asset}</p>
                                <p className="mb-1">
                                    <span className="font-semibold">Profit:</span>{" "}
                                    <span className={Number(trade.profit) >= 0 ? "text-green-400" : "text-red-400"}>
                                        {trade.profit ? `${Number(trade.profit).toFixed(2)}` : "0.00"}
                                    </span>
                                </p>
                                <p className="mb-4"><span className="font-semibold">Amount:</span> ${trade.amount}</p>

                                <div className="flex flex-wrap gap-2">
                                    <button
                                        onClick={async () => {
                                            await handleTradeAction(trade.id, "approve");
                                        }}
                                        className="flex-1 p-2 rounded-xl shadow-md text-white font-semibold transition bg-green-600 hover:bg-green-500"
                                        title="Approve"
                                    >
                                        <FiCheckCircle className="mx-auto" size={20} />
                                    </button>
                                    <button
                                        onClick={async () => {
                                            await handleTradeAction(trade.id, "decline");
                                        }}
                                        className="flex-1 p-2 rounded-xl shadow-md text-white font-semibold transition bg-red-600 hover:bg-red-500"
                                        title="Decline"
                                    >
                                        <FiXCircle className="mx-auto" size={20} />
                                    </button>

                                    <button
                                        onClick={() => handleEdit(trade)}
                                        className="px-4 py-2 rounded-xl shadow-md text-white font-semibold transition transform hover:scale-105 bg-blue-600 hover:bg-blue-500"
                                    >
                                        Edit
                                    </button>

                                    <button
                                        onClick={() => handleDelete(trade)}
                                        className="px-4 py-2 rounded-xl shadow-md text-white font-semibold transition transform hover:scale-105 bg-red-600 hover:bg-red-500"
                                    >
                                        Delete
                                    </button>
                                </div>
                            </div>
                        );
                    })}
                </div>
            </main>

            {/* Image Preview Modal */}
            {previewImage && (
                <div
                    className="fixed inset-0 bg-black/70 flex items-center justify-center z-50 p-4"
                    onClick={() => setPreviewImage(null)}
                >
                    <div className="relative max-w-4xl max-h-[90vh]">
                        <button
                            className="absolute -top-10 right-0 text-white hover:text-red-500 transition"
                            onClick={() => setPreviewImage(null)}
                        >
                            <FiX size={32} />
                        </button>
                        <img
                            src={previewImage}
                            alt="Proof Preview"
                            className="w-full h-full object-contain rounded-lg shadow-2xl bg-white dark:bg-gray-900"
                        />
                    </div>
                </div>
            )}

            {/* Add Trade Modal */}
            {showAddModal && (
                <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                    <div className={`${theme === "dark" ? "bg-gray-800" : "bg-white"} rounded-xl shadow-2xl max-w-2xl w-full max-h-[90vh] overflow-y-auto`}>
                        <div className="p-6">
                            <div className="flex justify-between items-center mb-4">
                                <h2 className={`text-2xl font-bold ${theme === "dark" ? "text-gray-100" : "text-gray-900"}`}>Add New Trade</h2>
                                <button
                                    onClick={() => setShowAddModal(false)}
                                    className={`text-2xl ${theme === "dark" ? "text-gray-400 hover:text-gray-200" : "text-gray-500 hover:text-gray-700"}`}
                                >
                                    ×
                                </button>
                            </div>
                            <form onSubmit={handleSubmitAdd}>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>User</label>
                                        <select
                                            value={formData.user_id}
                                            onChange={(e) => setFormData({ ...formData, user_id: e.target.value })}
                                            required
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                        >
                                            <option value="">Select User</option>
                                            {users.map((user) => (
                                                <option key={user.id} value={user.id}>{user.name}</option>
                                            ))}
                                        </select>
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Asset</label>
                                        <input
                                            type="text"
                                            value={formData.asset}
                                            onChange={(e) => setFormData({ ...formData, asset: e.target.value })}
                                            required
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="e.g., BTC, ETH"
                                        />
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Amount</label>
                                        <input
                                            type="number"
                                            step="0.01"
                                            value={formData.amount}
                                            onChange={(e) => setFormData({ ...formData, amount: e.target.value })}
                                            required
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="0.00"
                                        />
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Direction</label>
                                        <select
                                            value={formData.direction}
                                            onChange={(e) => setFormData({ ...formData, direction: e.target.value })}
                                            required
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                        >
                                            <option value="buy">Buy</option>
                                            <option value="sell">Sell</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Leverage</label>
                                        <input
                                            type="number"
                                            value={formData.leverage}
                                            onChange={(e) => setFormData({ ...formData, leverage: e.target.value })}
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="e.g., 10"
                                        />
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Status</label>
                                        <select
                                            value={formData.status}
                                            onChange={(e) => setFormData({ ...formData, status: e.target.value })}
                                            required
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                        >
                                            <option value="pending">Pending</option>
                                            <option value="approved">Approved</option>
                                            <option value="declined">Declined</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Profit</label>
                                        <input
                                            type="number"
                                            step="0.01"
                                            value={formData.profit}
                                            onChange={(e) => setFormData({ ...formData, profit: e.target.value })}
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="0.00"
                                        />
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Purchase Price</label>
                                        <input
                                            type="number"
                                            step="0.01"
                                            value={formData.purchase_price}
                                            onChange={(e) => setFormData({ ...formData, purchase_price: e.target.value })}
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="0.00"
                                        />
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Delivery Price</label>
                                        <input
                                            type="number"
                                            step="0.01"
                                            value={formData.delivery_price}
                                            onChange={(e) => setFormData({ ...formData, delivery_price: e.target.value })}
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="0.00"
                                        />
                                    </div>
                                </div>
                                <div className="flex gap-4 mt-6">
                                    <button
                                        type="submit"
                                        className="flex-1 px-6 py-3 bg-green-600 hover:bg-green-500 text-white font-semibold rounded-lg transition"
                                    >
                                        Add Trade
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => setShowAddModal(false)}
                                        className={`flex-1 px-6 py-3 font-semibold rounded-lg transition ${theme === "dark" ? "bg-gray-700 hover:bg-gray-600 text-gray-200" : "bg-gray-200 hover:bg-gray-300 text-gray-900"}`}
                                    >
                                        Cancel
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            )}

            {/* Edit Trade Modal */}
            {showEditModal && (
                <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                    <div className={`${theme === "dark" ? "bg-gray-800" : "bg-white"} rounded-xl shadow-2xl max-w-2xl w-full max-h-[90vh] overflow-y-auto`}>
                        <div className="p-6">
                            <div className="flex justify-between items-center mb-4">
                                <h2 className={`text-2xl font-bold ${theme === "dark" ? "text-gray-100" : "text-gray-900"}`}>Edit Trade</h2>
                                <button
                                    onClick={() => setShowEditModal(false)}
                                    className={`text-2xl ${theme === "dark" ? "text-gray-400 hover:text-gray-200" : "text-gray-500 hover:text-gray-700"}`}
                                >
                                    ×
                                </button>
                            </div>
                            <form onSubmit={handleSubmitEdit}>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>User</label>
                                        <select
                                            value={formData.user_id}
                                            onChange={(e) => setFormData({ ...formData, user_id: e.target.value })}
                                            required
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                        >
                                            <option value="">Select User</option>
                                            {users.map((user) => (
                                                <option key={user.id} value={user.id}>{user.name}</option>
                                            ))}
                                        </select>
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Asset</label>
                                        <input
                                            type="text"
                                            value={formData.asset}
                                            onChange={(e) => setFormData({ ...formData, asset: e.target.value })}
                                            required
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="e.g., BTC, ETH"
                                        />
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Amount</label>
                                        <input
                                            type="number"
                                            step="0.01"
                                            value={formData.amount}
                                            onChange={(e) => setFormData({ ...formData, amount: e.target.value })}
                                            required
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="0.00"
                                        />
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Direction</label>
                                        <select
                                            value={formData.direction}
                                            onChange={(e) => setFormData({ ...formData, direction: e.target.value })}
                                            required
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                        >
                                            <option value="buy">Buy</option>
                                            <option value="sell">Sell</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Leverage</label>
                                        <input
                                            type="number"
                                            value={formData.leverage}
                                            onChange={(e) => setFormData({ ...formData, leverage: e.target.value })}
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="e.g., 10"
                                        />
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Status</label>
                                        <select
                                            value={formData.status}
                                            onChange={(e) => setFormData({ ...formData, status: e.target.value })}
                                            required
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                        >
                                            <option value="pending">Pending</option>
                                            <option value="approved">Approved</option>
                                            <option value="declined">Declined</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Profit</label>
                                        <input
                                            type="number"
                                            step="0.01"
                                            value={formData.profit}
                                            onChange={(e) => setFormData({ ...formData, profit: e.target.value })}
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="0.00"
                                        />
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Purchase Price</label>
                                        <input
                                            type="number"
                                            step="0.01"
                                            value={formData.purchase_price}
                                            onChange={(e) => setFormData({ ...formData, purchase_price: e.target.value })}
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="0.00"
                                        />
                                    </div>
                                    <div>
                                        <label className={`block mb-2 font-semibold ${theme === "dark" ? "text-gray-200" : "text-gray-700"}`}>Delivery Price</label>
                                        <input
                                            type="number"
                                            step="0.01"
                                            value={formData.delivery_price}
                                            onChange={(e) => setFormData({ ...formData, delivery_price: e.target.value })}
                                            className={`w-full px-4 py-2 rounded-lg border ${theme === "dark" ? "bg-gray-700 border-gray-600 text-gray-200" : "bg-gray-100 border-gray-300 text-gray-900"}`}
                                            placeholder="0.00"
                                        />
                                    </div>
                                </div>
                                <div className="flex gap-4 mt-6">
                                    <button
                                        type="submit"
                                        className="flex-1 px-6 py-3 bg-blue-600 hover:bg-blue-500 text-white font-semibold rounded-lg transition"
                                    >
                                        Update Trade
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => setShowEditModal(false)}
                                        className={`flex-1 px-6 py-3 font-semibold rounded-lg transition ${theme === "dark" ? "bg-gray-700 hover:bg-gray-600 text-gray-200" : "bg-gray-200 hover:bg-gray-300 text-gray-900"}`}
                                    >
                                        Cancel
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            )}

            {/* Delete Confirmation Modal */}
            {showDeleteModal && (
                <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                    <div className={`${theme === "dark" ? "bg-gray-800" : "bg-white"} rounded-xl shadow-2xl max-w-md w-full`}>
                        <div className="p-6">
                            <h2 className={`text-2xl font-bold mb-4 ${theme === "dark" ? "text-gray-100" : "text-gray-900"}`}>Confirm Delete</h2>
                            <p className={`mb-6 ${theme === "dark" ? "text-gray-300" : "text-gray-700"}`}>
                                Are you sure you want to delete trade #{selectedTrade?.id}? This action cannot be undone.
                            </p>
                            <div className="flex gap-4">
                                <button
                                    onClick={handleConfirmDelete}
                                    className="flex-1 px-6 py-3 bg-red-600 hover:bg-red-500 text-white font-semibold rounded-lg transition"
                                >
                                    Delete
                                </button>
                                <button
                                    onClick={() => {
                                        setShowDeleteModal(false);
                                        setSelectedTrade(null);
                                    }}
                                    className={`flex-1 px-6 py-3 font-semibold rounded-lg transition ${theme === "dark" ? "bg-gray-700 hover:bg-gray-600 text-gray-200" : "bg-gray-200 hover:bg-gray-300 text-gray-900"}`}
                                >
                                    Cancel
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
}
