import { useState, useEffect, useRef } from "react";
import Navbar from "@/Components/Navbar";
import { router } from "@inertiajs/react";
import axios from "axios";
import { FaUser, FaComments, FaPaperPlane } from "react-icons/fa";

export default function AdminSupport({ auth }) {
    const user = auth?.user || {};
    const [theme, setTheme] = useState(localStorage.getItem("theme") || "light");
    const [chats, setChats] = useState([]);
    const [selectedChat, setSelectedChat] = useState(null);
    const [messages, setMessages] = useState([]);
    const [replyText, setReplyText] = useState("");
    const [loading, setLoading] = useState(false);
    const messagesEndRef = useRef(null);

    // Apply theme
    useEffect(() => {
        document.documentElement.classList.toggle("dark", theme === "dark");
        localStorage.setItem("theme", theme);
    }, [theme]);

    // Fetch Chats
    const fetchChats = async () => {
        try {
            const res = await axios.get('/api/admin/chats');
            setChats(res.data);
        } catch (error) {
            if (error.response?.status === 401) {
                window.location.href = "/login";
                return;
            }
            console.error("Fetch chats error:", error);
        }
    };

    // Initial Fetch & Polling for Chats List
    useEffect(() => {
        fetchChats();
        const interval = setInterval(fetchChats, 10000); // Poll list every 10s
        return () => clearInterval(interval);
    }, []);

    // Fetch Messages when Key Chat Selected
    const fetchMessages = async (chatId, shouldScroll = false) => {
        try {
            const res = await axios.get(`/api/admin/chat/${chatId}/messages`);
            setMessages(res.data);
            if (shouldScroll) {
                setTimeout(scrollToBottom, 100);
            }
        } catch (error) {
            if (error.response?.status === 401) {
                window.location.href = "/login";
                return;
            }
            console.error("Fetch messages error:", error);
        }
    };

    // Polling Messages for Active Chat
    useEffect(() => {
        if (selectedChat) {
            fetchMessages(selectedChat.id, true); // Initial load: scroll
            const interval = setInterval(() => fetchMessages(selectedChat.id, false), 3000); // Poll: DO NOT scroll
            return () => clearInterval(interval);
        }
    }, [selectedChat]);

    const scrollToBottom = () => {
        messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
    };

    const handleReply = async (e) => {
        e.preventDefault();
        if (!replyText.trim() || !selectedChat) return;

        setLoading(true);
        try {
            router.post(`/api/admin/chat/${selectedChat.id}/reply`, {
                message: replyText
            }, {
                preserveScroll: true,
                onSuccess: () => {
                    setReplyText("");
                    fetchMessages(selectedChat.id, true);
                }
            });
        } catch (error) {
            console.error("Reply error:", error);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className={`min-h-screen transition-colors duration-500 flex flex-col ${theme === "dark"
            ? "bg-gray-900 text-gray-100"
            : "bg-gray-50 text-gray-900"
            }`}>
            <Navbar user={user} />

            <div className="flex-1 max-w-7xl w-full mx-auto p-4 md:p-6 grid grid-cols-1 md:grid-cols-3 gap-6 h-[calc(100vh-80px)]">

                {/* Chat List Sidebar */}
                <div className={`rounded-2xl shadow-lg border overflow-hidden flex flex-col ${theme === "dark" ? "bg-gray-800 border-gray-700" : "bg-white border-gray-200"
                    }`}>
                    <div className="p-4 border-b border-gray-200 dark:border-gray-700 bg-blue-600 text-white flex justify-between items-center">
                        <h2 className="font-bold flex items-center space-x-2">
                            <FaComments /> <span>Active Conversations</span>
                        </h2>
                        <span className="bg-white/20 px-2 py-1 rounded text-xs">
                            {chats.length}
                        </span>
                    </div>
                    <div className="flex-1 overflow-y-auto p-2 space-y-2">
                        {chats.length === 0 ? (
                            <div className="text-center p-4 text-gray-500">No active chats.</div>
                        ) : (
                            chats.map(chat => (
                                <button
                                    key={chat.id}
                                    onClick={() => setSelectedChat(chat)}
                                    className={`w-full text-left p-3 rounded-lg transition flex items-center justify-between ${selectedChat?.id === chat.id
                                        ? "bg-blue-100 dark:bg-blue-900/50 border-blue-500 border"
                                        : "hover:bg-gray-100 dark:hover:bg-gray-700"
                                        }`}
                                >
                                    <div className="flex items-center space-x-3">
                                        <div className="bg-gradient-to-br from-indigo-500 to-purple-500 w-10 h-10 rounded-full flex items-center justify-center text-white font-bold">
                                            {chat.user?.name?.charAt(0) || "U"}
                                        </div>
                                        <div>
                                            <p className="font-semibold text-sm">{chat.user?.name || "Unknown User"}</p>
                                            <p className="text-xs text-gray-500 dark:text-gray-400 truncate w-32">
                                                {chat.last_message?.message || "No messages"}
                                            </p>
                                        </div>
                                    </div>
                                    {chat.unread_count > 0 && (
                                        <span className="bg-red-500 text-white text-xs font-bold w-5 h-5 flex items-center justify-center rounded-full">
                                            {chat.unread_count}
                                        </span>
                                    )}
                                </button>
                            ))
                        )}
                    </div>
                </div>

                {/* Chat Window */}
                <div className={`md:col-span-2 rounded-2xl shadow-lg border overflow-hidden flex flex-col ${theme === "dark" ? "bg-gray-800 border-gray-700" : "bg-white border-gray-200"
                    }`}>
                    {selectedChat ? (
                        <>
                            {/* Header */}
                            <div className="p-4 border-b border-gray-200 dark:border-gray-700 flex items-center space-x-3 bg-gray-50 dark:bg-gray-900/50">
                                <FaUser className="text-gray-500" />
                                <div>
                                    <h3 className="font-bold">{selectedChat.user?.name}</h3>
                                    <p className="text-xs text-gray-500">{selectedChat.user?.email}</p>
                                </div>
                            </div>

                            {/* Messages */}
                            <div className="flex-1 overflow-y-auto p-4 space-y-4 bg-gray-50 dark:bg-gray-900 scrollbar-thin">
                                {messages.map((msg, idx) => {
                                    const isAdmin = msg.sender_type === 'admin';
                                    const isBot = msg.sender_type === 'bot';
                                    return (
                                        <div key={idx} className={`flex ${isAdmin ? 'justify-end' : 'justify-start'}`}>
                                            <div className={`max-w-[80%] rounded-2xl px-4 py-3 shadow-md ${isAdmin
                                                ? 'bg-blue-600 text-white rounded-br-none'
                                                : isBot
                                                    ? 'bg-gray-200 dark:bg-gray-600 text-gray-800 dark:text-gray-200 border-l-4 border-blue-500'
                                                    : 'bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 rounded-bl-none border border-gray-200 dark:border-gray-600'
                                                }`}>
                                                {isBot && <span className="text-xs font-bold block mb-1 text-blue-600 dark:text-blue-400">🤖 Automated</span>}
                                                <p>{msg.message}</p>
                                                <span className={`text-[10px] block mt-1 opacity-70 ${isAdmin ? 'text-right' : 'text-left'}`}>
                                                    {new Date(msg.created_at).toLocaleString()}
                                                </span>
                                            </div>
                                        </div>
                                    );
                                })}
                                <div ref={messagesEndRef} />
                            </div>

                            {/* Input */}
                            <form onSubmit={handleReply} className="p-4 bg-white dark:bg-gray-800 border-t dark:border-gray-700 flex items-center space-x-3">
                                <input
                                    type="text"
                                    value={replyText}
                                    onChange={(e) => setReplyText(e.target.value)}
                                    placeholder="Type a reply..."
                                    className="flex-1 rounded-full border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:border-blue-500 focus:ring focus:ring-blue-200 dark:focus:ring-blue-900 px-4 py-2"
                                />
                                <button
                                    type="submit"
                                    disabled={loading || !replyText.trim()}
                                    className="bg-blue-600 hover:bg-blue-700 text-white p-3 rounded-full transition shadow-lg disabled:opacity-50"
                                >
                                    <FaPaperPlane />
                                </button>
                            </form>
                        </>
                    ) : (
                        <div className="flex-1 flex flex-col items-center justify-center text-gray-400 p-10 text-center">
                            <FaComments className="w-16 h-16 mb-4 opacity-20" />
                            <p className="text-xl font-medium">Select a conversation to start chatting</p>
                        </div>
                    )}
                </div>

            </div>
        </div>
    );
}
