import React, { useState } from "react";
import { Link, router } from "@inertiajs/react";
import { FiHome, FiGrid, FiDollarSign, FiMenu, FiX, FiMoon, FiSun, FiLogOut } from "react-icons/fi";

export default function Sidebar({ theme, setTheme }) {
    const [isOpen, setIsOpen] = useState(true); // Desktop sidebar toggle
    const [mobileOpen, setMobileOpen] = useState(false); // Mobile sidebar toggle

    const navLinks = [
        { href: "/", label: "Home", icon: <FiHome size={20} /> },
        { href: "/admin-trades", label: "Dashboard", icon: <FiGrid size={20} /> },
        { href: "/admin-wallet", label: "Wallet", icon: <FiDollarSign size={20} /> },
    ];

    return (
        <>
            {/* Mobile Header */}
            <div className="md:hidden flex justify-between items-center p-4 bg-white dark:bg-gray-900 shadow">
                <h2 className="text-xl font-bold text-gray-800 dark:text-white">Admin Panel</h2>
                <button onClick={() => setMobileOpen(true)} className="text-gray-800 dark:text-gray-200">
                    <FiMenu size={24} />
                </button>
            </div>

            {/* Mobile Sidebar Overlay */}
            {mobileOpen && (
                <div className="fixed inset-0 z-50 flex">
                    <div
                        className="fixed inset-0 bg-black/50"
                        onClick={() => setMobileOpen(false)}
                    ></div>
                    <div className="relative bg-white dark:bg-gray-900 w-64 p-4 transition-transform transform">
                        <button
                            onClick={() => setMobileOpen(false)}
                            className="absolute top-4 right-4 text-gray-800 dark:text-gray-200"
                        >
                            <FiX size={24} />
                        </button>

                        {/* Navigation */}
                        <nav className="flex flex-col mt-10 space-y-4 text-gray-700 dark:text-gray-300">
                            {navLinks.map((link) => (
                                <Link
                                    key={link.href}
                                    href={link.href}
                                    className="flex items-center px-4 py-2 rounded-md hover:bg-gray-100 dark:hover:bg-gray-800 transition"
                                >
                                    <span className="mr-3">{link.icon}</span>
                                    {link.label}
                                </Link>
                            ))}
                        </nav>

                        {/* Theme Toggle */}
                        <button
                            onClick={() => setTheme(theme === "light" ? "dark" : "light")}
                            className="w-full mt-6 px-4 py-2 rounded-md bg-gray-200 dark:bg-gray-800 text-gray-800 dark:text-gray-200 hover:bg-gray-300 dark:hover:bg-gray-700 transition flex items-center justify-center gap-2"
                        >
                            {theme === "light" ? <FiMoon /> : <FiSun />}
                            {theme === "light" ? "Dark Mode" : "Light Mode"}
                        </button>

                        {/* Logout */}
                        <Link
                            href="/logout"
                            method="post"
                            as="button"
                            className="w-full mt-4 px-4 py-2 rounded-md bg-red-500 text-white hover:bg-red-600 transition flex items-center justify-center gap-2"
                        >
                            <FiLogOut /> Logout
                        </Link>
                    </div>
                </div>
            )}

            {/* Desktop Sidebar */}
            <aside
                className={`hidden md:flex flex-col transition-all duration-300 bg-white dark:bg-gray-900 shadow-xl min-h-screen p-4 border-r dark:border-gray-700 ${isOpen ? "w-64" : "w-20"
                    }`}
            >
                <button
                    onClick={() => setIsOpen(!isOpen)}
                    className="w-full flex justify-end mb-4 text-gray-700 dark:text-gray-300"
                >
                    {isOpen ? "⬅" : "➡"}
                </button>

                <h2
                    className={`text-2xl font-bold mb-6 text-gray-800 dark:text-white transition-all duration-300 ${isOpen ? "opacity-100" : "opacity-0 hidden"
                        }`}
                >
                    Admin Panel
                </h2>

                <nav className="flex flex-col space-y-2 text-gray-700 dark:text-gray-300">
                    {navLinks.map((link) => (
                        <Link
                            key={link.href}
                            href={link.href}
                            className="flex items-center px-4 py-2 rounded-md hover:bg-gray-100 dark:hover:bg-gray-800 transition"
                        >
                            {link.icon}
                            <span className={`ml-3 ${isOpen ? "inline" : "hidden"}`}>{link.label}</span>
                        </Link>
                    ))}
                </nav>

                <div className="my-6 border-t dark:border-gray-700"></div>

                <button
                    onClick={() => setTheme(theme === "light" ? "dark" : "light")}
                    className="w-full px-4 py-2 rounded-md bg-gray-200 dark:bg-gray-800 text-gray-800 dark:text-gray-200 hover:bg-gray-300 dark:hover:bg-gray-700 transition flex items-center justify-center gap-2"
                >
                    {theme === "light" ? <FiMoon /> : <FiSun />}
                    <span className={`${isOpen ? "inline" : "hidden"}`}>
                        {theme === "light" ? "Dark Mode" : "Light Mode"}
                    </span>
                </button>

                <Link
                    href="/logout"
                    method="post"
                    as="button"
                    className="w-full mt-4 px-4 py-2 rounded-md bg-red-500 text-white hover:bg-red-600 transition flex items-center justify-center gap-2"
                >
                    <FiLogOut /> <span className={`${isOpen ? "inline" : "hidden"}`}>Logout</span>
                </Link>
            </aside>
        </>
    );
}
