import { useState, useEffect } from 'react';
import Navbar from '@/Components/Navbar';
import axios from 'axios';
import { FaCheckCircle, FaTimesCircle, FaDownload } from 'react-icons/fa';

export default function AdminKycRequests({ auth }) {
    const [requests, setRequests] = useState([]);
    const [loading, setLoading] = useState(true);

    const [confirmModal, setConfirmModal] = useState({ show: false, type: null, id: null });
    const [rejectionReason, setRejectionReason] = useState('');

    const [notification, setNotification] = useState({ message: "", type: "" });

    const showNotification = (message, type = "success") => {
        setNotification({ message, type });
        setTimeout(() => setNotification({ message: "", type: "" }), 3000);
    };

    useEffect(() => {
        fetchRequests();
    }, []);

    const fetchRequests = async () => {
        try {
            const res = await axios.get('/api/admin/kyc-requests');
            setRequests(res.data);
        } catch (error) {
            console.error("Failed to fetch KYC requests", error);
        } finally {
            setLoading(false);
        }
    };

    const openConfirmModal = (id, type) => {
        setConfirmModal({ show: true, type, id });
        setRejectionReason('');
    };

    const closeConfirmModal = () => {
        setConfirmModal({ show: false, type: null, id: null });
        setRejectionReason('');
    };

    const processStatusUpdate = async () => {
        const { id, type } = confirmModal;
        if (!id || !type) return;

        if (type === 'rejected' && !rejectionReason.trim()) {
            showNotification("Please enter a rejection reason.", "error");
            return;
        }

        try {
            await axios.post(`/api/admin/kyc/${id}/update`, {
                status: type,
                rejection_reason: type === 'rejected' ? rejectionReason : null
            });
            showNotification(`Request ${type} successfully.`);
            fetchRequests(); // Refresh list
            closeConfirmModal();
        } catch (error) {
            console.error("Update failed", error);
            showNotification("Failed to update status.", "error");
        }
    };

    return (
        <div className="min-h-screen bg-gray-100 dark:bg-gray-900">
            <Navbar user={auth.user} />

            {/* Notification Toast */}
            {notification.message && (
                <div className={`fixed top-20 left-1/2 -translate-x-1/2 px-6 py-4 rounded-xl shadow-2xl z-50 text-white font-bold animate-slide-in ${notification.type === 'success' ? 'bg-green-500' : 'bg-red-500'
                    }`}>
                    {notification.message}
                </div>
            )}

            <div className="max-w-7xl mx-auto py-10 px-4 sm:px-6 lg:px-8">
                <h1 className="text-3xl font-bold text-gray-800 dark:text-white mb-8">KYC Verification Requests</h1>

                {loading ? (
                    <p className="text-gray-500">Loading requests...</p>
                ) : requests.length === 0 ? (
                    <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow text-center">
                        <p className="text-gray-500">No pending KYC requests found.</p>
                    </div>
                ) : (
                    <div className="grid grid-cols-1 gap-6">
                        {requests.map((req) => (
                            <div key={req.id} className="bg-white dark:bg-gray-800 rounded-lg shadow-lg overflow-hidden flex flex-col md:flex-row">
                                {/* Images */}
                                <div className="md:w-1/3 bg-gray-200 dark:bg-gray-700 p-4 flex flex-col gap-4 items-center justify-center">
                                    <div className="w-full">
                                        <p className="text-xs font-bold text-gray-500 uppercase mb-1">Front Image</p>
                                        <a href={`/storage/${req.front_image_path}`} target="_blank" rel="noopener noreferrer">
                                            <img src={`/storage/${req.front_image_path}`} alt="Front" className="w-full h-40 object-cover rounded hover:opacity-90 transition cursor-zoom-in" />
                                        </a>
                                    </div>
                                    {req.back_image_path && (
                                        <div className="w-full">
                                            <p className="text-xs font-bold text-gray-500 uppercase mb-1">Back Image</p>
                                            <a href={`/storage/${req.back_image_path}`} target="_blank" rel="noopener noreferrer">
                                                <img src={`/storage/${req.back_image_path}`} alt="Back" className="w-full h-40 object-cover rounded hover:opacity-90 transition cursor-zoom-in" />
                                            </a>
                                        </div>
                                    )}
                                </div>

                                {/* Details Comparison */}
                                <div className="p-6 flex-1 flex flex-col">
                                    <div className="flex justify-between items-start mb-6">
                                        <h3 className="text-xl font-bold text-gray-900 dark:text-white flex items-center gap-2">
                                            Request #{req.id}
                                            <span className="text-sm font-normal text-gray-500">({new Date(req.created_at).toLocaleString()})</span>
                                        </h3>
                                        <span className="bg-blue-100 text-blue-800 text-xs px-3 py-1 rounded-full font-bold uppercase tracking-wide border border-blue-200">
                                            {req.document_type.replace('_', ' ')}
                                        </span>
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                                        {/* Column 1: Registered Info */}
                                        <div className="bg-gray-50 dark:bg-gray-900 p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                                            <h4 className="font-bold text-gray-700 dark:text-gray-300 mb-3 border-b pb-2 uppercase text-xs tracking-wider">
                                                🖥️ Registered Account Info
                                            </h4>
                                            <div className="space-y-2 text-sm">
                                                <p><span className="text-gray-500">Name:</span> <span className="font-medium text-gray-900 dark:text-white">{req.user.name}</span></p>
                                                <p><span className="text-gray-500">Email:</span> <span className="font-medium text-gray-900 dark:text-white">{req.user.email}</span></p>
                                                <p><span className="text-gray-500">Joined:</span> <span className="text-gray-900 dark:text-white">{new Date(req.user.created_at).toLocaleDateString()}</span></p>
                                                <p><span className="text-gray-500">ID:</span> <span className="text-gray-900 dark:text-white">{req.user.id}</span></p>
                                            </div>
                                        </div>

                                        {/* Column 2: Submitted KYC Data */}
                                        <div className="bg-blue-50 dark:bg-blue-900/20 p-4 rounded-lg border border-blue-100 dark:border-blue-800">
                                            <h4 className="font-bold text-blue-800 dark:text-blue-300 mb-3 border-b border-blue-200 pb-2 uppercase text-xs tracking-wider">
                                                📄 Submitted Document Details
                                            </h4>
                                            <div className="space-y-2 text-sm">
                                                <div className="grid grid-cols-2 gap-x-4">
                                                    <p><span className="text-blue-600 dark:text-blue-400">First Name:</span> <span className="font-medium text-gray-900 dark:text-white">{req.first_name || '-'}</span></p>
                                                    <p><span className="text-blue-600 dark:text-blue-400">Last Name:</span> <span className="font-medium text-gray-900 dark:text-white">{req.last_name || '-'}</span></p>
                                                </div>
                                                <p><span className="text-blue-600 dark:text-blue-400">Email:</span> <span className="font-medium text-gray-900 dark:text-white">{req.email || '-'}</span></p>
                                                <div className="grid grid-cols-2 gap-x-4">
                                                    <p><span className="text-blue-600 dark:text-blue-400">Phone:</span> <span className="text-gray-900 dark:text-white">{req.phone || '-'}</span></p>
                                                    <p><span className="text-blue-600 dark:text-blue-400">DOB:</span> <span className="text-gray-900 dark:text-white">{req.dob || '-'}</span></p>
                                                </div>
                                                <div className="mt-2 pt-2 border-t border-blue-200 dark:border-blue-800/50">
                                                    <p><span className="text-blue-600 dark:text-blue-400">Address:</span> <span className="text-gray-900 dark:text-white">{req.address_line || '-'}</span></p>
                                                    <p><span className="text-blue-600 dark:text-blue-400">City/State:</span> <span className="text-gray-900 dark:text-white">{req.city}, {req.state}</span></p>
                                                    <p><span className="text-blue-600 dark:text-blue-400">Nationality:</span> <span className="font-bold text-gray-900 dark:text-white">{req.nationality || '-'}</span></p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    {/* Actions */}
                                    <div className="mt-auto flex space-x-4 pt-4 border-t dark:border-gray-700">
                                        <button
                                            onClick={() => openConfirmModal(req.id, 'approved')}
                                            className="flex-1 bg-green-600 hover:bg-green-700 text-white font-bold py-2 px-4 rounded flex items-center justify-center space-x-2 transition"
                                        >
                                            <FaCheckCircle /> <span>Approve</span>
                                        </button>
                                        <button
                                            onClick={() => openConfirmModal(req.id, 'rejected')}
                                            className="flex-1 bg-red-600 hover:bg-red-700 text-white font-bold py-2 px-4 rounded flex items-center justify-center space-x-2 transition"
                                        >
                                            <FaTimesCircle /> <span>Reject</span>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>
                )}

                {/* Confirm Action Modal */}
                {confirmModal.show && (
                    <div className="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black bg-opacity-50 backdrop-blur-sm">
                        <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-2xl max-w-sm w-full p-6 border border-gray-100 dark:border-gray-700 animate-fade-in">
                            <h3 className={`text-xl font-bold mb-4 ${confirmModal.type === 'approved' ? 'text-green-600' : 'text-red-600'}`}>
                                {confirmModal.type === 'approved' ? 'Approve Request' : 'Reject Request'}
                            </h3>

                            <p className="text-gray-600 dark:text-gray-300 mb-4">
                                {confirmModal.type === 'approved'
                                    ? "Are you sure you want to approve this user's KYC documents?"
                                    : "Are you sure you want to reject this request? Please provide a reason below."}
                            </p>

                            {confirmModal.type === 'rejected' && (
                                <textarea
                                    className="w-full p-3 border rounded-lg bg-gray-50 dark:bg-gray-900 dark:border-gray-700 dark:text-white focus:ring-2 focus:ring-red-500 mb-4 text-sm"
                                    rows="3"
                                    placeholder="Enter reason for rejection..."
                                    value={rejectionReason}
                                    onChange={(e) => setRejectionReason(e.target.value)}
                                ></textarea>
                            )}

                            <div className="flex space-x-3">
                                <button
                                    onClick={closeConfirmModal}
                                    className="flex-1 py-2 px-4 bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-white font-semibold rounded-lg hover:bg-gray-300 transition"
                                >
                                    Cancel
                                </button>
                                <button
                                    onClick={processStatusUpdate}
                                    className={`flex-1 py-2 px-4 text-white font-semibold rounded-lg shadow-lg transition ${confirmModal.type === 'approved'
                                        ? 'bg-green-600 hover:bg-green-700'
                                        : 'bg-red-600 hover:bg-red-700'
                                        }`}
                                >
                                    Confirm
                                </button>
                            </div>
                        </div>
                    </div>
                )}
            </div>

            <style>{`
                @keyframes fade-in {
                    from { opacity: 0; transform: translateY(10px); }
                    to { opacity: 1; transform: translateY(0); }
                }
                .animate-fade-in {
                    animation: fade-in 0.2s ease-out;
                }
            `}</style>
        </div>
    );
}
