import { useState, useEffect } from "react";
import Navbar from "@/Components/Navbar";
import { Link, router } from "@inertiajs/react";

export default function AdminHome({ auth, users, trades, transactions }) {
    const user = auth?.user || {};
    const [theme, setTheme] = useState(localStorage.getItem("theme") || "light");

    // Apply theme globally and sync with Navbar
    useEffect(() => {
        const handleThemeChange = (e) => {
            setTheme(e.detail);
        };
        window.addEventListener('themeChanged', handleThemeChange);

        document.documentElement.classList.toggle("dark", theme === "dark");
        localStorage.setItem("theme", theme);

        return () => window.removeEventListener('themeChanged', handleThemeChange);
    }, [theme]);

    // Polling for real-time updates
    useEffect(() => {
        const interval = setInterval(() => {
            router.reload({ only: ['trades', 'transactions', 'users'], preserveScroll: true });
        }, 3000); // Poll every 3 seconds

        return () => clearInterval(interval);
    }, []);

    return (
        <div className={`min-h-screen transition-colors duration-500 ${theme === "dark"
            ? "bg-gray-900 text-gray-100"
            : "bg-gradient-to-r from-purple-100 via-pink-50 to-white text-gray-900"
            }`}>
            {/* Navbar */}
            <Navbar user={user} />

            {/* Main Content - No Sidebar */}
            <main className="p-4 md:p-6 w-full max-w-7xl mx-auto">
                <h1 className="text-3xl font-bold mb-4">Admin Dashboard</h1>
                <p className="mb-6 text-opacity-80">Welcome, {user.name}! Control center for Coinsova.</p>

                {/* Quick Links Grid */}
                <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-4 mb-10">
                    <Link
                        href="/admin-wallet"
                        className={`p-4 rounded-2xl shadow-lg transition transform hover:scale-105 ${theme === "dark"
                            ? "bg-gray-800 hover:bg-gray-700"
                            : "bg-white/80 hover:bg-white/90 backdrop-blur-md"
                            }`}
                    >
                        <h2 className="text-lg font-bold mb-1">Wallet</h2>
                        <p className="text-sm opacity-70">Deposits & Withdrawals</p>
                    </Link>

                    <Link
                        href="/admin-trades"
                        className={`p-4 rounded-2xl shadow-lg transition transform hover:scale-105 ${theme === "dark"
                            ? "bg-gray-800 hover:bg-gray-700"
                            : "bg-white/80 hover:bg-white/90 backdrop-blur-md"
                            }`}
                    >
                        <h2 className="text-lg font-bold mb-1">Trades</h2>
                        <p className="text-sm opacity-70">User Trade Activity</p>
                    </Link>

                    <Link
                        href="/admin-kyc"
                        className={`p-4 rounded-2xl shadow-lg transition transform hover:scale-105 ${theme === "dark"
                            ? "bg-gray-800 hover:bg-gray-700"
                            : "bg-white/80 hover:bg-white/90 backdrop-blur-md"
                            }`}
                    >
                        <h2 className="text-lg font-bold mb-1">KYC</h2>
                        <p className="text-sm opacity-70">Identity Verifications</p>
                    </Link>

                    <Link
                        href="/admin-support"
                        className={`p-4 rounded-2xl shadow-lg transition transform hover:scale-105 ${theme === "dark"
                            ? "bg-gray-800 hover:bg-gray-700"
                            : "bg-white/80 hover:bg-white/90 backdrop-blur-md"
                            }`}
                    >
                        <h2 className="text-lg font-bold mb-1">Support</h2>
                        <p className="text-sm opacity-70">Customer Service</p>
                    </Link>

                    <Link
                        href="/admin-home"
                        className={`p-4 rounded-2xl shadow-lg transition transform hover:scale-105 ${theme === "dark"
                            ? "bg-gray-800 hover:bg-gray-700 border-2 border-purple-500"
                            : "bg-white/80 hover:bg-white/90 border-2 border-purple-400"
                            }`}
                    >
                        <h2 className="text-lg font-bold mb-1">Dashboard</h2>
                        <p className="text-sm opacity-70">System Overview</p>
                    </Link>
                </div>

                {/* Grid for Trades and Transactions - NOW ON TOP */}
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 mb-10">
                    {/* Recent Trades */}
                    <div className={`p-6 rounded-2xl shadow-lg backdrop-blur-md ${theme === "dark" ? "bg-gray-800" : "bg-white/80"
                        }`}>
                        <div className="flex justify-between items-center mb-4">
                            <h2 className="text-2xl font-bold">Recent Trades</h2>
                            <Link href="/admin-trades" className="text-purple-600 hover:underline text-sm font-bold">View All</Link>
                        </div>
                        <div className="overflow-x-auto">
                            <table className="w-full text-left border-collapse">
                                <thead>
                                    <tr className="border-b dark:border-gray-700">
                                        <th className="p-3">User</th>
                                        <th className="p-3">Asset</th>
                                        <th className="p-3">Amount</th>
                                        <th className="p-3">Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {trades && trades.length > 0 ? (
                                        trades.map((t) => (
                                            <tr key={t.id} className="border-b dark:border-gray-700 hover:bg-gray-700/50 transition">
                                                <td className="p-3">{t.user?.name || 'Unknown'}</td>
                                                <td className="p-3">{t.asset}</td>
                                                <td className="p-3">${t.amount}</td>
                                                <td className="p-3">
                                                    <span className={`px-2 py-1 rounded text-xs font-bold uppercase ${t.status === 'approved' ? 'bg-green-600 text-white' :
                                                        t.status === 'declined' ? 'bg-red-600 text-white' :
                                                            'bg-yellow-600 text-white'
                                                        }`}>
                                                        {t.status}
                                                    </span>
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan="4" className="p-4 text-center text-gray-500 italic">No recent trades.</td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>
                    </div>

                    {/* Recent Transactions */}
                    <div className={`p-6 rounded-2xl shadow-lg backdrop-blur-md ${theme === "dark" ? "bg-gray-800" : "bg-white/80"
                        }`}>
                        <div className="flex justify-between items-center mb-4">
                            <h2 className="text-2xl font-bold">Recent Transactions</h2>
                            <Link href="/admin-wallet" className="text-purple-600 hover:underline text-sm font-bold">View All</Link>
                        </div>
                        <div className="overflow-x-auto">
                            <table className="w-full text-left border-collapse">
                                <thead>
                                    <tr className="border-b dark:border-gray-700">
                                        <th className="p-3">User</th>
                                        <th className="p-3">Type</th>
                                        <th className="p-3">Amount</th>
                                        <th className="p-3">Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {transactions && transactions.length > 0 ? (
                                        transactions.map((tx) => (
                                            <tr key={tx.id} className="border-b dark:border-gray-700 hover:bg-gray-700/50 transition">
                                                <td className="p-3">{tx.user?.name || 'Unknown'}</td>
                                                <td className="p-3 capitalize">{tx.type}</td>
                                                <td className="p-3">${tx.amount}</td>
                                                <td className="p-3">
                                                    <span className={`px-2 py-1 rounded text-xs font-bold uppercase ${tx.status === 'approved' ? 'bg-green-600 text-white' :
                                                        tx.status === 'declined' ? 'bg-red-600 text-white' :
                                                            'bg-yellow-600 text-white'
                                                        }`}>
                                                        {tx.status}
                                                    </span>
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan="4" className="p-4 text-center text-gray-500 italic">No recent transactions.</td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                {/* Users List Section - NOW AT BOTTOM */}
                <div className={`p-6 rounded-2xl shadow-lg backdrop-blur-md mb-8 ${theme === "dark" ? "bg-gray-800" : "bg-white/80"
                    }`}>
                    <h2 className="text-2xl font-bold mb-4">Registered Users</h2>
                    <div className="overflow-x-auto">
                        <table className="w-full text-left border-collapse">
                            <thead>
                                <tr className="border-b dark:border-gray-700 text-lg">
                                    <th className="p-3">ID</th>
                                    <th className="p-3">Name</th>
                                    <th className="p-3">Email</th>
                                    <th className="p-3">Role</th>
                                    <th className="p-3">Joined</th>
                                </tr>
                            </thead>
                            <tbody>
                                {users && users.length > 0 ? (
                                    users.map((u) => (
                                        <tr key={u.id} className="border-b dark:border-gray-700 hover:bg-gray-700/50 transition">
                                            <td className="p-3 text-purple-400 font-mono">#{u.id}</td>
                                            <td className="p-3 font-medium">{u.name}</td>
                                            <td className="p-3">{u.email}</td>
                                            <td className="p-3">
                                                <span className={`px-2 py-1 rounded text-sm font-bold ${Number(u.role) === 1 ? 'bg-purple-600 text-white' : 'bg-blue-600 text-white'
                                                    }`}>
                                                    {Number(u.role) === 1 ? 'Admin' : 'User'}
                                                </span>
                                            </td>
                                            <td className="p-3">{new Date(u.created_at).toLocaleDateString()}</td>
                                        </tr>
                                    ))
                                ) : (
                                    <tr>
                                        <td colSpan="5" className="p-4 text-center text-gray-500 italic">No users found.</td>
                                    </tr>
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>
            </main>
        </div>
    );
}
