import { useState, useEffect, useRef } from 'react';
import { Link } from '@inertiajs/react';
import { FaUserCircle, FaSun, FaMoon } from 'react-icons/fa';
import ChatWidget from '@/Components/ChatWidget';
import { motion, AnimatePresence } from 'framer-motion';

export default function Navbar({ user }) {
  // ... (keep existing state and effects logic) ...
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [profileMenuOpen, setProfileMenuOpen] = useState(false);
  const [darkMode, setDarkMode] = useState(false);
  const dropdownRef = useRef(null);

  const currentUser = user || null;

  // ... (keep existing useEffects) ...
  // Load saved theme on initial load
  useEffect(() => {
    const savedTheme = localStorage.getItem("theme");

    if (savedTheme === "dark") {
      setDarkMode(true);
      document.documentElement.classList.add("dark");
    } else {
      setDarkMode(false);
      document.documentElement.classList.remove("dark");
    }
  }, []);

  // Save theme + apply it when changed
  useEffect(() => {
    const newTheme = darkMode ? "dark" : "light";
    if (darkMode) {
      document.documentElement.classList.add("dark");
      localStorage.setItem("theme", "dark");
    } else {
      document.documentElement.classList.remove("dark");
      localStorage.setItem("theme", "light");
    }
    // Dispatch custom event for real-time sync across components
    window.dispatchEvent(new CustomEvent('themeChanged', { detail: newTheme }));
  }, [darkMode]);

  // Close dropdown when clicking outside
  useEffect(() => {
    const handleClickOutside = (event) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target)) {
        setProfileMenuOpen(false);
      }
    };
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);


  const handleGuestClick = (e) => {
    e.preventDefault();
    window.location.href = '/login';
  };

  const links = [
    { name: 'Home', href: '/' },
    { name: 'Dashboard', href: '/user-dashboard' },
    { name: 'Wallet', href: '/wallet' },
    { name: 'About Us', href: '/about-us' },
    { name: 'Contact Us', href: '/contact-us' },
  ];

  const adminLinks = [
    { name: 'Admin Home', href: '/admin-home' },
    { name: 'Admin Trades', href: '/admin-trades' },
    { name: 'Admin Wallet', href: '/admin-wallet' },
    { name: 'Admin Support', href: '/admin-support' },
    { name: 'KYC Requests', href: '/admin-kyc' },
  ];

  return (
    <>
      <nav className="bg-gradient-to-r from-purple-600 via-blue-600 to-indigo-600 text-white shadow-xl sticky top-0 z-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between h-16 items-center">

            {/* Logo */}
            <div className="flex-shrink-0 text-3xl font-extrabold tracking-wide hover:scale-105 transition-transform">
              <Link href="/">Coinsova</Link>
            </div>

            {/* Desktop Links */}
            <div className="hidden md:flex space-x-6 font-medium text-lg">
              {/* User Links - Hide for Admin */}
              {(!currentUser || Number(currentUser.role) !== 1) && links.map(link => (
                <Link
                  key={link.name}
                  href={currentUser ? link.href : '#'}
                  onClick={currentUser ? null : handleGuestClick}
                  className="hover:text-yellow-400 transition duration-300 transform hover:scale-105"
                >
                  {link.name}
                </Link>
              ))}

              {/* Admin Links - Show only for Admin */}
              {currentUser && Number(currentUser.role) === 1 && adminLinks.map(link => (
                <Link
                  key={link.name}
                  href={link.href}
                  className="hover:text-yellow-400 transition duration-300 transform hover:scale-105"
                >
                  {link.name}
                </Link>
              ))}
            </div>

            {/* Profile + Theme */}
            <div className="hidden md:flex items-center space-x-4">
              {currentUser ? (
                <div className="relative" ref={dropdownRef}>
                  <button
                    onClick={() => setProfileMenuOpen(!profileMenuOpen)}
                    className="flex items-center space-x-2 bg-blue-800 px-4 py-2 rounded-full hover:bg-blue-900 transition shadow-md"
                  >
                    <FaUserCircle className="w-8 h-8 text-white" />
                    <span className="font-medium">{currentUser.name}</span>
                  </button>

                  {profileMenuOpen && (
                    <div className="absolute right-0 mt-2 w-48 bg-white dark:bg-gray-800 text-gray-800 dark:text-gray-200 rounded-lg shadow-xl py-2 z-50">
                      <Link
                        href="/profile"
                        className="block px-4 py-2 hover:bg-gray-100 dark:hover:bg-gray-700 transition"
                        onClick={() => setProfileMenuOpen(false)}
                      >
                        Profile
                      </Link>

                      <Link
                        href="/logout"
                        method="post"
                        as="button"
                        className="w-full text-left block px-4 py-3 bg-gradient-to-r from-red-500 to-rose-600 text-white font-bold rounded-lg hover:shadow-lg hover:shadow-red-500/50 transform hover:scale-105 transition-all duration-300 text-center"
                        onClick={() => setProfileMenuOpen(false)}
                      >
                        Logout
                      </Link>

                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          setDarkMode(!darkMode);
                        }}
                        className="w-full text-left px-4 py-2 hover:bg-gray-100 dark:hover:bg-gray-700 transition flex items-center space-x-2"
                      >
                        {darkMode ? <FaMoon /> : <FaSun />}
                        <span>{darkMode ? 'Dark' : 'Light'} Mode</span>
                      </button>
                    </div>
                  )}
                </div>
              ) : (
                <Link
                  href="/login"
                  className="px-6 py-3 bg-gradient-to-r from-yellow-400 to-orange-500 text-gray-900 font-bold rounded-xl shadow-lg hover:shadow-yellow-500/50 transform hover:scale-105 hover:-translate-y-1 transition-all duration-300"
                >
                  Login
                </Link>
              )}
            </div>

            {/* Mobile Menu Button */}
            <div className="md:hidden flex items-center">
              <button
                onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
                className="p-2 rounded-md focus:outline-none focus:ring-2 focus:ring-white"
              >
                <svg
                  className="h-6 w-6"
                  xmlns="http://www.w3.org/2000/svg"
                  fill="none"
                  viewBox="0 0 24 24"
                  stroke="currentColor"
                >
                  {mobileMenuOpen ? (
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                  ) : (
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                  )}
                </svg>
              </button>
            </div>
          </div>
        </div>

        {/* Mobile Menu */}
        <AnimatePresence>
          {mobileMenuOpen && (
            <motion.div
              initial={{ opacity: 0, y: -20 }}
              animate={{ opacity: 1, y: 0 }}
              exit={{ opacity: 0, y: -20 }}
              transition={{ duration: 0.2 }}
              className="md:hidden absolute top-16 left-0 w-full bg-gradient-to-r from-purple-600 via-blue-600 to-indigo-600 px-2 pt-2 pb-4 space-y-1 shadow-2xl z-50 border-t border-purple-500/30"
            >
              {(!currentUser || Number(currentUser.role) !== 1) && links.map(link => (
                <Link
                  key={link.name}
                  href={currentUser ? link.href : '#'}
                  onClick={currentUser ? () => setMobileMenuOpen(false) : handleGuestClick}
                  className="block px-3 py-2 rounded hover:bg-blue-700 transition font-medium"
                >
                  {link.name}
                </Link>
              ))}

              {currentUser && Number(currentUser.role) === 1 && adminLinks.map(link => (
                <Link
                  key={link.name}
                  href={link.href}
                  onClick={() => setMobileMenuOpen(false)}
                  className="block px-3 py-2 rounded hover:bg-blue-700 transition font-medium text-yellow-300"
                >
                  {link.name}
                </Link>
              ))}

              {!currentUser && (
                <Link
                  href="/login"
                  className="block mx-2 my-2 px-6 py-3 bg-gradient-to-r from-yellow-400 to-orange-500 text-gray-900 font-bold rounded-xl shadow-lg hover:shadow-yellow-500/50 transform hover:scale-105 transition-all duration-300 text-center"
                >
                  Login
                </Link>
              )}

              {currentUser && (
                <>
                  <Link
                    href="/profile"
                    className="block px-3 py-2 rounded hover:bg-blue-700 transition font-medium"
                    onClick={() => setMobileMenuOpen(false)}
                  >
                    Profile
                  </Link>

                  <button
                    onClick={() => setDarkMode(!darkMode)}
                    className="w-full text-left px-3 py-2 rounded hover:bg-blue-700 transition flex items-center space-x-2 font-medium"
                  >
                    {darkMode ? <FaMoon /> : <FaSun />}
                    <span>{darkMode ? 'Dark' : 'Light'} Mode</span>
                  </button>

                  <Link
                    href="/logout"
                    method="post"
                    as="button"
                    className="w-full text-left block mx-2 my-2 px-6 py-3 bg-gradient-to-r from-red-500 to-rose-600 text-white font-bold rounded-xl shadow-lg hover:shadow-red-500/50 transform hover:scale-105 transition-all duration-300 text-center"
                    onClick={() => setMobileMenuOpen(false)}
                  >
                    Logout
                  </Link>
                </>
              )}
            </motion.div>
          )}
        </AnimatePresence>
      </nav>
      {/* Global Customer Support Chat (Hide for Admin) */}
      {(!currentUser || Number(currentUser.role) !== 1) && <ChatWidget user={currentUser} />}
    </>
  );
}
