import { useState, useRef } from 'react';
import { useForm } from '@inertiajs/react';
import { FaCheckCircle, FaExclamationTriangle } from 'react-icons/fa';

export default function KycForm({ status, rejectionReason }) {
    const { data, setData, post, processing, errors } = useForm({
        first_name: '',
        last_name: '',
        email: '',
        phone: '',
        dob: '',
        address_line: '',
        city: '',
        state: '',
        nationality: '',
        document_type: 'id_card',
        front_image: null,
        back_image: null,
        details_confirmed: false,
    });

    const [previewFront, setPreviewFront] = useState(null);
    const [previewBack, setPreviewBack] = useState(null);
    const [showConfirmModal, setShowConfirmModal] = useState(false);

    const handleFileChange = (e, field) => {
        const file = e.target.files[0];
        setData(field, file);

        if (file) {
            const reader = new FileReader();
            reader.onloadend = () => {
                if (field === 'front_image') setPreviewFront(reader.result);
                if (field === 'back_image') setPreviewBack(reader.result);
            };
            reader.readAsDataURL(file);
        }
    };

    const confirmSubmit = () => {
        post(route('kyc.store'), {
            onSuccess: () => setShowConfirmModal(false),
            onError: () => setShowConfirmModal(false),
        });
    };

    if (status === 'pending') {
        return (
            <div className="bg-yellow-50 dark:bg-yellow-900/20 border-l-4 border-yellow-500 p-6 rounded shadow-sm mb-6 animate-fade-in">
                <div className="flex items-center">
                    <FaExclamationTriangle className="text-yellow-500 w-6 h-6 mr-4" />
                    <div>
                        <h3 className="text-lg font-bold text-yellow-800 dark:text-yellow-300">Verification Pending</h3>
                        <p className="text-sm text-yellow-700 dark:text-yellow-400 mt-1">
                            Your documents are currently under review by our team. This process typically takes 24-48 hours.
                        </p>
                    </div>
                </div>
            </div>
        );
    }

    if (status === 'approved') {
        return (
            <div className="bg-green-50 dark:bg-green-900/20 border-l-4 border-green-500 p-6 rounded shadow-sm mb-6 animate-fade-in">
                <div className="flex items-center">
                    <FaCheckCircle className="text-green-500 w-6 h-6 mr-4" />
                    <div>
                        <h3 className="text-lg font-bold text-green-800 dark:text-green-300">Account Verified</h3>
                        <p className="text-sm text-green-700 dark:text-green-400 mt-1">
                            Your identity has been successfully verified. You now have full access to all platform features.
                        </p>
                    </div>
                </div>
            </div>
        );
    }

    return (
        <section className="bg-white dark:bg-gray-800 shadow-xl rounded-2xl p-8 max-w-4xl mx-auto">
            <header className="mb-8 border-b pb-4 dark:border-gray-700">
                <h2 className="text-2xl font-bold text-gray-900 dark:text-gray-100">Identity Verification</h2>
                <p className="mt-2 text-gray-600 dark:text-gray-400">
                    To comply with regulations and ensure the security of your account, please complete the verification form below.
                </p>
            </header>

            {status === 'rejected' && (
                <div className="mb-6 bg-red-50 dark:bg-red-900/20 border border-red-400 text-red-700 dark:text-red-400 px-4 py-3 rounded relative">
                    <strong className="font-bold">Verification Rejected:</strong>
                    <span className="block sm:inline"> {rejectionReason}</span>
                    <p className="text-sm mt-1">Please correct the issue and re-submit.</p>
                </div>
            )}

            <form onSubmit={(e) => { e.preventDefault(); setShowConfirmModal(true); }} className="space-y-8">

                {/* Personal Details */}
                <div>
                    <h3 className="text-xl font-semibold text-gray-800 dark:text-white mb-4 border-l-4 border-blue-500 pl-3">Personal Details</h3>
                    <p className="text-sm text-gray-500 dark:text-gray-400 mb-4">Your simple personal information required for identification.</p>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">First Name *</label>
                            <input type="text" value={data.first_name} onChange={e => setData('first_name', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 dark:bg-gray-700 dark:border-gray-600 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
                            {errors.first_name && <p className="text-red-500 text-xs mt-1">{errors.first_name}</p>}
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">Last Name *</label>
                            <input type="text" value={data.last_name} onChange={e => setData('last_name', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 dark:bg-gray-700 dark:border-gray-600 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
                            {errors.last_name && <p className="text-red-500 text-xs mt-1">{errors.last_name}</p>}
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">Email *</label>
                            <input type="email" value={data.email} onChange={e => setData('email', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 dark:bg-gray-700 dark:border-gray-600 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
                            {errors.email && <p className="text-red-500 text-xs mt-1">{errors.email}</p>}
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">Phone Number *</label>
                            <input type="tel" value={data.phone} onChange={e => setData('phone', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 dark:bg-gray-700 dark:border-gray-600 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
                            {errors.phone && <p className="text-red-500 text-xs mt-1">{errors.phone}</p>}
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">Date of Birth *</label>
                            <input type="date" value={data.dob} onChange={e => setData('dob', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 dark:bg-gray-700 dark:border-gray-600 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
                            {errors.dob && <p className="text-red-500 text-xs mt-1">{errors.dob}</p>}
                        </div>
                    </div>
                </div>

                {/* Address Details */}
                <div>
                    <h3 className="text-xl font-semibold text-gray-800 dark:text-white mb-4 border-l-4 border-blue-500 pl-3">Your Address</h3>
                    <p className="text-sm text-gray-500 dark:text-gray-400 mb-4">Your simple location information required for identification.</p>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div className="md:col-span-2">
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">Address Line *</label>
                            <input type="text" value={data.address_line} onChange={e => setData('address_line', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 dark:bg-gray-700 dark:border-gray-600 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
                            {errors.address_line && <p className="text-red-500 text-xs mt-1">{errors.address_line}</p>}
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">City *</label>
                            <input type="text" value={data.city} onChange={e => setData('city', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 dark:bg-gray-700 dark:border-gray-600 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
                            {errors.city && <p className="text-red-500 text-xs mt-1">{errors.city}</p>}
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">State *</label>
                            <input type="text" value={data.state} onChange={e => setData('state', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 dark:bg-gray-700 dark:border-gray-600 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
                            {errors.state && <p className="text-red-500 text-xs mt-1">{errors.state}</p>}
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">Nationality *</label>
                            <input type="text" value={data.nationality} onChange={e => setData('nationality', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 dark:bg-gray-700 dark:border-gray-600 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500" required />
                            {errors.nationality && <p className="text-red-500 text-xs mt-1">{errors.nationality}</p>}
                        </div>
                    </div>
                </div>

                {/* Document Upload */}
                <div>
                    <h3 className="text-xl font-semibold text-gray-800 dark:text-white mb-4 border-l-4 border-blue-500 pl-3">Document Upload</h3>
                    <p className="text-sm text-gray-500 dark:text-gray-400 mb-4">Your simple personal document required for identification.</p>

                    <div className="bg-blue-50 dark:bg-blue-900/20 p-4 rounded-lg mb-6 border border-blue-100 dark:border-blue-800">
                        <h4 className="font-semibold text-blue-800 dark:text-blue-300 mb-2">Requirements:</h4>
                        <ul className="list-disc list-inside text-sm text-blue-700 dark:text-blue-400 space-y-1">
                            <li>Chosen credential must not have expired.</li>
                            <li>Document should be in good condition and clearly visible.</li>
                            <li>Make sure that there is no light glare on the document.</li>
                        </ul>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">Document Type</label>
                            <select value={data.document_type} onChange={e => setData('document_type', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 dark:bg-gray-700 dark:border-gray-600 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500">
                                <option value="id_card">ID Card</option>
                                <option value="passport">Passport</option>
                                <option value="driver_license">Driver's License</option>
                            </select>
                        </div>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mt-4">
                        <div className="border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg p-6 text-center hover:bg-gray-50 dark:hover:bg-gray-900 transition">
                            <label className="cursor-pointer block">
                                <span className="block font-medium text-gray-700 dark:text-gray-300 mb-2">Upload Front Side *</span>
                                <input type="file" accept="image/*" onChange={e => handleFileChange(e, 'front_image')} className="hidden" />
                                {previewFront ? (
                                    <img src={previewFront} alt="Front Preview" className="h-40 mx-auto object-cover rounded shadow" />
                                ) : (
                                    <div className="text-gray-400">
                                        <div className="text-4xl mb-2">📄</div>
                                        <span className="text-sm">Click to upload</span>
                                    </div>
                                )}
                            </label>
                            {errors.front_image && <p className="text-red-500 text-xs mt-2">{errors.front_image}</p>}
                        </div>

                        <div className="border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg p-6 text-center hover:bg-gray-50 dark:hover:bg-gray-900 transition">
                            <label className="cursor-pointer block">
                                <span className="block font-medium text-gray-700 dark:text-gray-300 mb-2">Upload Back Side (Optional)</span>
                                <input type="file" accept="image/*" onChange={e => handleFileChange(e, 'back_image')} className="hidden" />
                                {previewBack ? (
                                    <img src={previewBack} alt="Back Preview" className="h-40 mx-auto object-cover rounded shadow" />
                                ) : (
                                    <div className="text-gray-400">
                                        <div className="text-4xl mb-2">📄</div>
                                        <span className="text-sm">Click to upload</span>
                                    </div>
                                )}
                            </label>
                            {errors.back_image && <p className="text-red-500 text-xs mt-2">{errors.back_image}</p>}
                        </div>
                    </div>
                </div>

                {/* Confirmation Checkbox */}
                <div className="flex items-start">
                    <div className="flex items-center h-5">
                        <input
                            id="details_confirmed"
                            type="checkbox"
                            checked={data.details_confirmed}
                            onChange={e => setData('details_confirmed', e.target.checked)}
                            className="focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300 rounded"
                            required
                        />
                    </div>
                    <div className="ml-3 text-sm">
                        <label htmlFor="details_confirmed" className="font-medium text-gray-700 dark:text-gray-300">
                            All The Information I Have Entered Is Correct.
                        </label>
                        <p className="text-gray-500 dark:text-gray-400">I confirm that the submitted documents are valid and belong to me.</p>
                    </div>
                </div>
                {errors.details_confirmed && <p className="text-red-500 text-xs mt-1 ml-7">{errors.details_confirmed}</p>}


                <button
                    type="submit"
                    disabled={processing}
                    className="w-full bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white font-bold py-4 rounded-xl shadow-lg transform hover:scale-[1.02] transition-all disabled:opacity-50 disabled:cursor-not-allowed"
                >
                    {processing ? 'Submitting Verification...' : 'Submit Verification'}
                </button>
            </form>

            {/* Confirmation Modal */}
            {showConfirmModal && (
                <div className="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black bg-opacity-50 backdrop-blur-sm animate-fade-in">
                    <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-2xl max-w-md w-full p-6 border border-gray-100 dark:border-gray-700">
                        <h3 className="text-xl font-bold text-gray-900 dark:text-white mb-4">Confirm Submission</h3>
                        <p className="text-gray-600 dark:text-gray-300 mb-6">
                            Are you sure you want to submit your details? You <strong>cannot edit</strong> these details once submitted.
                        </p>

                        <div className="bg-gray-50 dark:bg-gray-900 p-4 rounded-lg mb-6 text-sm space-y-2">
                            <p><strong className="text-gray-700 dark:text-gray-400">Name:</strong> {data.first_name} {data.last_name}</p>
                            <p><strong className="text-gray-700 dark:text-gray-400">Email:</strong> {data.email}</p>
                            <p><strong className="text-gray-700 dark:text-gray-400">Nationality:</strong> {data.nationality}</p>
                        </div>

                        <div className="flex space-x-3">
                            <button
                                onClick={() => setShowConfirmModal(false)}
                                className="flex-1 py-3 px-4 bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-white font-semibold rounded-xl hover:bg-gray-300 dark:hover:bg-gray-600 transition"
                            >
                                Cancel
                            </button>
                            <button
                                onClick={confirmSubmit}
                                className="flex-1 py-3 px-4 bg-blue-600 text-white font-semibold rounded-xl hover:bg-blue-700 shadow-lg transition"
                            >
                                Confirm & Submit
                            </button>
                        </div>
                    </div>
                </div>
            )}
        </section>
    );
}
