import { useState, useEffect, useRef } from 'react';
import axios from 'axios';
import { FaComments, FaTimes, FaPaperPlane, FaRobot } from 'react-icons/fa';

export default function ChatWidget({ user }) {
    const [isOpen, setIsOpen] = useState(false);
    const [messages, setMessages] = useState([]);
    const [newMessage, setNewMessage] = useState('');
    const [loading, setLoading] = useState(false);
    const messagesEndRef = useRef(null);
    const [hasInitialized, setHasInitialized] = useState(false);

    // Scroll to bottom
    const scrollToBottom = () => {
        messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
    };

    // Fetch messages
    const fetchChat = async () => {
        if (!user) return;
        try {
            const res = await axios.get('/chat/sync');
            setMessages(res.data.messages);
            setHasInitialized(true);
        } catch (error) {
            console.error("Chat sync error:", error);
        }
    };

    // Poll for new messages when open
    useEffect(() => {
        if (isOpen && user) {
            fetchChat();
            const interval = setInterval(fetchChat, 5000); // Poll every 5s
            return () => clearInterval(interval);
        }
    }, [isOpen, user]);

    const [isAtBottom, setIsAtBottom] = useState(true);

    // Initial scroll on open
    useEffect(() => {
        if (isOpen) {
            scrollToBottom();
        }
    }, [isOpen]);

    // Scroll to bottom on new messages ONLY if user is already at bottom
    useEffect(() => {
        if (isOpen && isAtBottom) {
            scrollToBottom();
        }
    }, [messages, isOpen, isAtBottom]);

    const handleScroll = (e) => {
        const { scrollTop, scrollHeight, clientHeight } = e.target;
        // Check if user is near bottom (within 50px)
        const atBottom = scrollHeight - scrollTop - clientHeight < 50;
        setIsAtBottom(atBottom);
    };

    const handleSend = async (e) => {
        e.preventDefault();
        if (!newMessage.trim()) return;

        // Force scroll to bottom when user sends a message
        setIsAtBottom(true);
        setTimeout(scrollToBottom, 50);

        const tempMsg = {
            id: Date.now(), // temporary ID
            sender_type: 'user',
            message: newMessage,
            created_at: new Date().toISOString(),
        };

        // Optimistic UI update
        setMessages(prev => [...prev, tempMsg]);
        setNewMessage('');
        setLoading(true);

        try {
            const res = await axios.post('/chat/send', { message: tempMsg.message });
            // Replace messages with server response (handles bot reply too)
            if (res.data.success) {
                // We'll refetch or merge. Simpler to refetch or use response.
                // The server returns the NEW messages created (user msg + potential bot msg).
                // But we need the whole history or careful merging. 
                // Let's just re-sync to be safe and consistent.
                fetchChat();
            }
        } catch (error) {
            console.error("Send error:", error);
        } finally {
            setLoading(false);
        }
    };

    if (!user) return null; // Don't show if not logged in

    return (
        <div className="fixed bottom-6 right-6 z-50">
            {/* Chat Window */}
            {isOpen && (
                <div className="mb-4 w-80 md:w-96 bg-white dark:bg-gray-800 rounded-2xl shadow-2xl border border-gray-200 dark:border-gray-700 overflow-hidden flex flex-col h-[500px]">
                    {/* Header */}
                    <div className="bg-gradient-to-r from-blue-600 to-indigo-600 p-4 flex justify-between items-center text-white">
                        <div className="flex items-center space-x-2">
                            <FaComments className="w-5 h-5" />
                            <span className="font-bold">Support Chat</span>
                        </div>
                        <button onClick={() => setIsOpen(false)} className="hover:bg-blue-700 p-1 rounded transition">
                            <FaTimes />
                        </button>
                    </div>

                    {/* Messages Area */}
                    <div
                        className="flex-1 overflow-y-auto p-4 space-y-4 bg-gray-50 dark:bg-gray-900 scrollbar-thin"
                        onScroll={handleScroll}
                    >
                        {messages.length === 0 && hasInitialized ? (
                            <div className="text-center text-gray-500 mt-10 text-sm">
                                <p>How can we help you today?</p>
                            </div>
                        ) : null}

                        {messages.map((msg, idx) => {
                            const isUser = msg.sender_type === 'user';
                            const isBot = msg.sender_type === 'bot';

                            // Format bot messages with proper line breaks and styling
                            const formatMessage = (text) => {
                                if (!isBot) return text;

                                // Split by newlines and render each line
                                return text.split('\n').map((line, i) => {
                                    // Handle bold text **text**
                                    const parts = line.split(/(\*\*.*?\*\*)/g);
                                    const formattedLine = parts.map((part, j) => {
                                        if (part.startsWith('**') && part.endsWith('**')) {
                                            return <strong key={j}>{part.slice(2, -2)}</strong>;
                                        }
                                        return part;
                                    });

                                    return <div key={i} className={line.trim() === '' ? 'h-2' : ''}>{formattedLine}</div>;
                                });
                            };

                            return (
                                <div key={idx} className={`flex ${isUser ? 'justify-end' : 'justify-start'}`}>
                                    <div className={`max-w-[85%] rounded-2xl px-4 py-3 text-sm shadow-sm ${isUser
                                        ? 'bg-blue-600 text-white rounded-br-none'
                                        : 'bg-white dark:bg-gray-700 text-gray-800 dark:text-gray-200 rounded-bl-none border border-gray-200 dark:border-gray-600'
                                        }`}>
                                        {isBot && (
                                            <div className="flex items-center space-x-1 mb-2 text-xs text-blue-500 dark:text-blue-400 font-bold uppercase tracking-wider">
                                                <FaRobot /> <span>AI Assistant</span>
                                            </div>
                                        )}
                                        <div className="whitespace-pre-wrap leading-relaxed">
                                            {formatMessage(msg.message)}
                                        </div>
                                        <span className={`text-[10px] block mt-2 opacity-70 ${isUser ? 'text-right' : 'text-left'}`}>
                                            {new Date(msg.created_at).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                                        </span>
                                    </div>
                                </div>
                            );
                        })}
                        <div ref={messagesEndRef} />
                    </div>

                    {/* Input Area */}
                    <form onSubmit={handleSend} className="p-3 bg-white dark:bg-gray-800 border-t dark:border-gray-700 flex items-center space-x-2">
                        <input
                            type="text"
                            value={newMessage}
                            onChange={(e) => setNewMessage(e.target.value)}
                            placeholder="Type your message..."
                            className="flex-1 rounded-full border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:border-blue-500 focus:ring focus:ring-blue-200 dark:focus:ring-blue-900 text-sm px-4 py-2"
                        />
                        <button
                            type="submit"
                            disabled={loading || !newMessage.trim()}
                            className="bg-blue-600 hover:bg-blue-700 text-white p-2.5 rounded-full transition shadow disabled:opacity-50 disabled:cursor-not-allowed"
                        >
                            <FaPaperPlane className="w-4 h-4 transform -rotate-45 translate-x-0.5" />
                        </button>
                    </form>
                </div>
            )}

            {/* Floating Button */}
            {!isOpen && (
                <button
                    onClick={() => setIsOpen(true)}
                    className="bg-blue-600 hover:bg-blue-700 text-white p-4 rounded-full shadow-lg hover:shadow-blue-500/50 transition-all transform hover:scale-110 animate-bounce-slow"
                >
                    <FaComments className="w-8 h-8" />
                </button>
            )}
        </div>
    );
}
