<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class VerificationCode extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'code',
        'type',
        'expires_at',
        'verified_at',
    ];

    protected $casts = [
        'expires_at' => 'datetime',
        'verified_at' => 'datetime',
    ];

    /**
     * Relationship with User
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Generate a random 6-digit code
     */
    public static function generateCode()
    {
        return str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
    }

    /**
     * Check if code is expired
     */
    public function isExpired()
    {
        return $this->expires_at->isPast();
    }

    /**
     * Check if code is verified
     */
    public function isVerified()
    {
        return !is_null($this->verified_at);
    }

    /**
     * Mark code as verified
     */
    public function markAsVerified()
    {
        $this->verified_at = now();
        $this->save();
    }

    /**
     * Create a new verification code for a user
     */
    public static function createForUser($userId, $type, $expiresInMinutes = 10)
    {
        // Delete old codes of the same type for this user
        static::where('user_id', $userId)
            ->where('type', $type)
            ->delete();

        return static::create([
            'user_id' => $userId,
            'code' => static::generateCode(),
            'type' => $type,
            'expires_at' => now()->addMinutes($expiresInMinutes),
        ]);
    }
}
