<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\WalletTransaction;
use App\Models\User;
use Illuminate\Support\Facades\Auth;

class WalletTransactionController extends Controller
{
    public function store(Request $request)
    {
        \Log::info('=== WALLET TRANSACTION STORE START ===', [
            'user_id' => Auth::id(),
            'request_data' => $request->all(),
        ]);

        $rules = [
            'crypto' => 'required|string',
            'amount' => 'required|numeric|min:0.00000001',
            'type' => 'required|in:deposit,withdraw',
            'proof' => 'nullable|file|mimes:jpg,jpeg,png,pdf',
        ];

        if ($request->type === 'withdraw') {
            $rules['wallet_address'] = 'required|string|min:10';
        }

        $request->validate($rules);

        $user = Auth::user();
        $crypto = strtoupper($request->crypto);

        \Log::info('Validation passed', [
            'user_id' => $user->id,
            'crypto' => $crypto,
            'amount' => $request->amount,
            'type' => $request->type,
            'wallet_address' => $request->wallet_address,
        ]);

        // For withdrawals, check if user has sufficient balance
        if ($request->type === 'withdraw') {
            $availableBalance = $user->getAvailableBalance($crypto);

            \Log::info('Withdrawal balance check', [
                'user_id' => $user->id,
                'crypto' => $crypto,
                'available_balance' => $availableBalance,
                'requested_amount' => $request->amount,
            ]);

            if ($request->amount > $availableBalance) {
                \Log::warning('Insufficient balance for withdrawal');
                return redirect()->back()->withErrors([
                    'amount' => 'Insufficient balance. Your available ' . $crypto . ' balance is ' . $availableBalance
                ]);
            }
        }

        $proofPath = $request->file('proof') ? $request->file('proof')->store('proofs', 'public') : null;

        \Log::info('File upload processed', ['proof_path' => $proofPath]);

        $transaction = WalletTransaction::create([
            'user_id' => $user->id,
            'crypto' => $crypto,
            'amount' => $request->amount,
            'type' => $request->type,
            'proof' => $proofPath,
            'status' => 'pending',  // Pending - admin must approve
            'wallet_address' => $request->wallet_address,
        ]);

        \Log::info('=== TRANSACTION CREATED ===', [
            'transaction_id' => $transaction->id,
            'user_id' => $user->id,
            'crypto' => $crypto,
            'amount' => $request->amount,
            'type' => $request->type,
            'status' => 'pending',
        ]);

        return redirect()->back()->with('success', 'Transaction request submitted!');
    }

    // Get user's wallet and transaction history
    public function userWallet()
    {
        $user = Auth::user()->fresh(['kycVerification']);

        \Log::info('UserWallet Request', [
            'user_id' => $user->id,
            'user_email' => $user->email,
        ]);

        // Calculate balances dynamically from approved transactions
        $cryptos = ['btc', 'eth', 'sol', 'bnb', 'xrp', 'ada', 'doge', 'trx', 'link', 'dot', 'matic', 'ltc', 'avax', 'atom', 'etc', 'usdt', 'usdc'];
        $wallet = [];

        foreach ($cryptos as $crypto) {
            $wallet[strtoupper($crypto)] = $user->getAvailableBalance($crypto);
        }

        $transactions = WalletTransaction::where('user_id', $user->id)
            ->whereIn('type', ['deposit', 'withdraw'])
            ->where('source', 'user')
            ->orderBy('created_at', 'desc')
            ->get();

        \Log::info('UserWallet Data', [
            'wallet' => $wallet,
            'transactions_count' => $transactions->count(),
            'transactions' => $transactions->toArray(),
        ]);

        return inertia('Wallet', [
            'auth' => ['user' => $user],
            'wallet' => $wallet,
            'trades' => $transactions,
        ]);
    }

    // Admin views all transactions
    public function index()
    {
        $transactions = WalletTransaction::with('user')
            ->where('source', 'user') // ONLY show user-initiated requests
            ->latest()
            ->get()
            ->map(function ($transaction) {
            // Generate proper URL for proof images
            if ($transaction->proof) {
                // Handle both old format (proofs/filename) and new format (proofs/filename)
                // If proof doesn't start with 'proofs/', it's likely in the old format
                $proofPath = $transaction->proof;
                if (!str_starts_with($proofPath, 'proofs/')) {
                    // Old format - might be in storage/app/proofs, need to check if file exists in public
                    // Try to find it in public storage first
                    if (file_exists(storage_path('app/public/' . $proofPath))) {
                        $proofPath = $proofPath;
                    } else {
                        // File might be in old location, but we'll try public storage anyway
                        $proofPath = 'proofs/' . basename($proofPath);
                    }
                }
                $transaction->proof_url = asset('storage/' . $proofPath);
            } else {
                $transaction->proof_url = null;
            }
            return $transaction;
        });
        
        return inertia('Admin/AdminWallet', [
            'transactions' => $transactions,
            'users' => User::select('id', 'name', 'email')->orderBy('name')->get()
        ]);
    }

    // Admin create transaction
    public function adminStore(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'crypto' => 'required|string',
            'amount' => 'required|numeric',
            'type' => 'required|in:deposit,withdraw',
            'status' => 'required|in:pending,approved,declined',
        ]);

        WalletTransaction::create([
            'user_id' => $validated['user_id'],
            'crypto' => strtoupper($validated['crypto']),
            'amount' => $validated['amount'],
            'type' => $validated['type'],
            'status' => $validated['status'],
        ]);

        return redirect()->back()->with('success', 'Transaction created successfully');
    }

    // Admin update transaction
    public function adminUpdate(Request $request, $id)
    {
        $transaction = WalletTransaction::findOrFail($id);
        
        $validated = $request->validate([
            'amount' => 'required|numeric',
            'status' => 'required|in:pending,approved,declined',
            'crypto' => 'required|string',
             // 'type' is usually fixed but can allow edit if needed. Let's allow minimal edits.
        ]);

        $transaction->update([
            'amount' => $validated['amount'],
            'status' => $validated['status'],
            'crypto' => strtoupper($validated['crypto']),
        ]);

        return redirect()->back()->with('success', 'Transaction updated successfully');                 
    }

    // Admin destroy transaction
    public function adminDestroy($id)
    {
        $transaction = WalletTransaction::findOrFail($id);
        $transaction->delete();
        return redirect()->back()->with('success', 'Transaction deleted successfully');
    }

    // Admin approves transaction
    public function approve($id)
    {
        $transaction = WalletTransaction::findOrFail($id);

        if ($transaction->status === 'pending') {
            $transaction->status = 'approved';
            $transaction->save();
        }

        return redirect()->back()->with('success', 'Transaction approved!');
    }

    // Admin declines transaction
    public function decline($id)
    {
        $transaction = WalletTransaction::findOrFail($id);

        if ($transaction->status === 'pending') {
            $transaction->status = 'declined';
            $transaction->save();
        }

        return redirect()->back()->with('error', 'Transaction declined.');
    }
    // Get user's current wallet balance (JSON for polling)
    public function getBalance()
    {
        $user = Auth::user();
        $cryptos = ['btc', 'eth', 'sol', 'bnb', 'xrp', 'ada', 'doge', 'trx', 'link', 'dot', 'matic', 'ltc', 'avax', 'atom', 'etc', 'usdt', 'usdc'];
        $wallet = [];

        foreach ($cryptos as $crypto) {
            $wallet[strtoupper($crypto)] = $user->getAvailableBalance($crypto);
        }

        return response()->json(['wallet' => $wallet]);
    }
}
