<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Auth;
use App\Models\Trade;
use App\Models\WalletTransaction;
use Carbon\Carbon;

class TradeController extends Controller
{
    // User Dashboard
    public function dashboard()
    {
        $user = Auth::user();

        // Initialize wallet with available balances (Total - Locked)
        $cryptos = ['BTC', 'ETH', 'SOL', 'BNB', 'XRP', 'ADA', 'DOGE', 'TRX', 'LINK', 'DOT', 'MATIC', 'LTC', 'AVAX', 'ATOM', 'ETC', 'USDT', 'USDC'];
        $wallet = [];
        foreach ($cryptos as $crypto) {
            $wallet[$crypto] = $user->getAvailableBalance($crypto);
        }

        // Fetch user's trades
        $trades = $user->trades()->latest()->get();

        return Inertia::render('Dashboard', [
            'auth' => ['user' => $user],
            'wallet' => $wallet,
            'trades' => $trades,
        ]);
    }

    // Place trade (buy/sell)
    public function placeTrade(Request $request)
    {
        $request->validate([
            'crypto'   => 'required|string',
            'amount'   => 'required|numeric|min:1',
            'leverage' => 'required|string',
            'expiry'   => 'required|string',
            'type'     => 'required|in:buy,sell',
        ]);

        $user = Auth::user();
        $coin = strtoupper($request->crypto);

        // Check available balance (considering active trades and pending withdrawals)
        $availableBalance = $user->getAvailableBalance($coin);
        if ($request->amount > $availableBalance) {
            $errorMsg = "Insufficient balance! You attempted to trade $request->amount $coin, but your available balance is only $availableBalance $coin. This trade cannot be made.";
            
            if ($request->wantsJson()) {
                return response()->json(['message' => $errorMsg], 422);
            }

            return redirect()->back()->withErrors(['amount' => $errorMsg]);
        }
        
        // Map string leverage to numeric value

        // Map string leverage to numeric value
        $leverageMap = [
            '1:10'  => 10,
            '1:35'  => 35,
            '1:87'  => 87,
            '1:165' => 165,
            '1:250' => 250,
            '1:365' => 365,
        ];
        $numericLeverage = $leverageMap[$request->leverage] ?? 1;

        // Calculate expiry timestamp
        // Input is in seconds (e.g., 60, 120, 3600)
        $seconds = (int) $request->expiry;
        if ($seconds <= 0) $seconds = 60; // Default to 1 minute
        
        $expiryAt = now()->addSeconds($seconds);

        // Fetch current market price from CoinGecko
        $coinMap = [
            'BTC' => 'bitcoin', 'ETH' => 'ethereum', 'SOL' => 'solana',
            'BNB' => 'binancecoin', 'XRP' => 'ripple', 'ADA' => 'cardano',
            'DOGE' => 'dogecoin', 'TRX' => 'tron', 'LINK' => 'chainlink',
            'DOT' => 'polkadot', 'MATIC' => 'matic-network', 'LTC' => 'litecoin',
            'AVAX' => 'avalanche-2', 'ATOM' => 'cosmos', 'ETC' => 'ethereum-classic',
        ];
        
        $coinId = $coinMap[$coin] ?? 'bitcoin';
        $purchasePrice = 0;
        
        try {
            $response = \Illuminate\Support\Facades\Http::withoutVerifying()
                ->get("https://api.coingecko.com/api/v3/simple/price", [
                    'ids' => $coinId,
                    'vs_currencies' => 'usd'
                ]);
            
            if ($response->successful() && isset($response->json()[$coinId]['usd'])) {
                $purchasePrice = $response->json()[$coinId]['usd'];
            }
        } catch (\Exception $e) {
            // If API fails, use 0 or handle error
            \Log::error('Failed to fetch market price: ' . $e->getMessage());
        }

        // Create trade
        $trade = Trade::create([
            'user_id'   => $user->id,
            'crypto'    => $coin,
            'asset'     => $coin,
            'amount'    => $request->amount,
            'leverage'  => $numericLeverage,
            'expiry_at' => $expiryAt,
            'direction' => $request->type,
            'status'    => 'pending',
            'purchase_price' => $purchasePrice,
        ]);

        // Don't create withdrawal transaction - balance is not deducted on placement
        // Balance will only be updated when trade settles with profit/loss

        return redirect()->back()->with('success', 'Trade placed successfully!')->with('trade', $trade)->with('current_price', $purchasePrice);
    }

    // Fetch latest trades (for real-time updates)
    public function latestTrades()
    {
        $user = Auth::user();

        // Force settlement of expired trades to ensure real-time updates match frontend timer
        // This fixes the issue where balance reverts if the scheduled job hasn't run yet
        \Illuminate\Support\Facades\Artisan::call('trades:settle');
        
        $trades = $user->trades()->latest()->get();

        $formatted = $trades->map(function($trade) {
            return [
                'id'          => $trade->id,
                'asset'       => $trade->asset,
                'amount'      => $trade->amount,
                'leverage'    => $trade->leverage,
                'expiry_at'   => $trade->expiry_at,
                'direction'   => $trade->direction,
                'status'      => $trade->status,
                'profit'      => $trade->profit,
                'purchase_price' => $trade->purchase_price,
                'delivery_price' => $trade->delivery_price,
                'created_at'  => $trade->created_at,
            ];
        });

        return response()->json(['trades' => $formatted]);
    }
}
