<?php

namespace App\Http\Controllers;

use App\Models\Chat;
use App\Models\ChatMessage;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Auth;

class ChatController extends Controller
{
    // -------------------------
    // User Methods
    // -------------------------

    // Get or Create the current user's active chat
    public function getUserChat()
    {
        $user = Auth::user();
        if (!$user) return response()->json(['error' => 'Unauthorized'], 401);

        // Find existing open chat or create new
        $chat = Chat::firstOrCreate(
            ['user_id' => $user->id, 'status' => 'open']
        );

        // Fetch messages
        $messages = $chat->messages()->with('chat')->oldest()->get();

        return response()->json([
            'chat' => $chat,
            'messages' => $messages
        ]);
    }

    // User sends a message
    public function sendMessage(Request $request)
    {
        $request->validate(['message' => 'required|string']);
        $user = Auth::user();

        $chat = Chat::firstOrCreate(
            ['user_id' => $user->id, 'status' => 'open']
        );

        // Save User Message
        $userMsg = $chat->messages()->create([
            'sender_type' => 'user',
            'message' => $request->message,
            'is_read' => false,
        ]);

        $responseMessages = [$userMsg];

        // Check for Bot Auto-Reply based on keywords
        $botReplyContent = $this->generateBotResponse($request->message);
        
        if ($botReplyContent) {
            $botMsg = $chat->messages()->create([
                'sender_type' => 'bot',
                'message' => $botReplyContent,
                'is_read' => true,
            ]);
            $responseMessages[] = $botMsg;
        }

        return response()->json([
            'success' => true,
            'messages' => $responseMessages
        ]);
    }

    // Helper: Generate Bot Response based on Keywords
    private function generateBotResponse($message)
    {
        $msg = strtolower(trim($message));
        $user = Auth::user();

        // Get user's wallet balances
        $cryptos = ['BTC', 'ETH', 'SOL', 'XRP', 'DOGE', 'USDT', 'USDC'];
        $balances = [];
        $totalBalanceUSDT = 0;

        foreach ($cryptos as $crypto) {
            $balance = $user->getAvailableBalance($crypto);
            $balances[$crypto] = $balance;
            $totalBalanceUSDT += $balance;
        }

        // ===== BALANCE QUERIES =====
        if (preg_match('/\b(balance|wallet|total|how much|my money|my funds)\b/i', $msg)) {
            $balanceDetails = "💰 **Your Current Balances:**\n\n";
            $balanceDetails .= "**Total Balance: $" . number_format($totalBalanceUSDT, 2) . " USDT**\n\n";
            $balanceDetails .= "Individual Balances:\n";
            
            foreach ($balances as $crypto => $amount) {
                if ($amount > 0) {
                    $balanceDetails .= "• {$crypto}: " . number_format($amount, 8) . " {$crypto}\n";
                }
            }
            
            if ($totalBalanceUSDT == 0) {
                $balanceDetails .= "\nYou currently have no funds. Make a deposit to get started!";
            }
            
            return $balanceDetails;
        }

        // ===== DEPOSIT QUERIES =====
        if (preg_match('/\b(deposit|add funds|add money|fund|top up|how to deposit)\b/i', $msg)) {
            return "💵 **How to Deposit:**\n\n1. Go to the **Wallet** page\n2. Select the cryptocurrency you want to deposit (BTC, ETH, SOL, etc.)\n3. Click the **DEPOSIT** tab\n4. Copy our wallet address or scan the QR code\n5. Send funds from your external wallet\n6. Upload payment proof (screenshot)\n7. Submit and wait for admin approval (usually within 24 hours)\n\n⚡ **Supported Cryptocurrencies:** BTC, ETH, SOL, XRP, DOGE, USDT, USDC\n\n💡 **Tip:** Always double-check the wallet address before sending!";
        }

        // ===== WITHDRAWAL QUERIES =====
        if (preg_match('/\b(withdraw|withdrawal|cash out|take out|send money|how to withdraw)\b/i', $msg)) {
            $kycStatus = $user->kyc_status;
            $kycWarning = "";
            
            if ($kycStatus !== 'approved') {
                $kycWarning = "\n\n⚠️ **Important:** Your KYC status is '{$kycStatus}'. You must complete KYC verification before withdrawing funds.";
            }
            
            return "💸 **How to Withdraw:**\n\n1. Go to the **Wallet** page\n2. Select the cryptocurrency you want to withdraw\n3. Click the **WITHDRAW** tab\n4. Enter the amount you want to withdraw\n5. **Paste your wallet address** (double-check it!)\n6. Submit the request\n7. Admin will process within 24 hours{$kycWarning}\n\n🔒 **Security Reminder:** Always verify your wallet address. Transactions cannot be reversed!";
        }

        // ===== TRADING QUERIES =====
        if (preg_match('/\b(trade|trading|how to trade|place trade|buy|sell|profit|loss)\b/i', $msg)) {
            return "📈 **How to Trade:**\n\n1. Go to the **Dashboard**\n2. Select your cryptocurrency pair (e.g., BTC/USDT)\n3. Choose your trade direction: **UP** (Call) or **DOWN** (Put)\n4. Enter your trade amount\n5. Select leverage (optional, increases potential profit/loss)\n6. Choose expiry time (when the trade closes)\n7. Click **Place Trade**\n\n💡 **Trading Tips:**\n• Start with small amounts to learn\n• Use leverage carefully - it amplifies both gains and losses\n• Monitor the market chart before placing trades\n• Your profit/loss is calculated based on the price movement\n\n📊 **Current Balance Available for Trading:** $" . number_format($totalBalanceUSDT, 2) . " USDT";
        }

        // ===== KYC/VERIFICATION QUERIES =====
        if (preg_match('/\b(kyc|verification|verify|identity|document|id|passport)\b/i', $msg)) {
            $kycStatus = $user->kyc_status;
            $statusEmoji = $kycStatus === 'approved' ? '✅' : ($kycStatus === 'pending' ? '⏳' : '❌');
            
            return "{$statusEmoji} **Your KYC Status: {$kycStatus}**\n\n**Why KYC is Required:**\n• Required for withdrawals\n• Ensures account security\n• Complies with regulations\n\n**How to Complete KYC:**\n1. Go to your **Profile** or **KYC** section\n2. Upload a government-issued ID (passport, driver's license)\n3. Upload a selfie holding your ID\n4. Submit for review\n5. Wait for admin approval (usually 24-48 hours)\n\n" . ($kycStatus === 'approved' ? "✅ Great! Your account is verified." : "⚠️ Please complete KYC to enable withdrawals.");
        }

        // ===== FEE QUERIES =====
        if (preg_match('/\b(fee|fees|charge|cost|commission)\b/i', $msg)) {
            return "💰 **Fee Structure:**\n\n**Deposits:** FREE ✅\n**Withdrawals:** Small network fee (varies by crypto)\n**Trading:** Included in spread (no separate fee)\n\n💡 **Note:** Blockchain network fees are paid to miners/validators, not to us. We keep our fees minimal to maximize your profits!";
        }

        // ===== ACCOUNT/PROFILE QUERIES =====
        if (preg_match('/\b(account|profile|settings|change password|update|email)\b/i', $msg)) {
            return "👤 **Account Management:**\n\n**Your Account Details:**\n• Name: {$user->name}\n• Email: {$user->email}\n• KYC Status: {$user->kyc_status}\n\n**What You Can Do:**\n• Update your profile information\n• Change your password\n• Complete KYC verification\n• View transaction history\n• Check your balance\n\nGo to the **Profile** section to manage your account settings.";
        }

        // ===== SECURITY QUERIES =====
        if (preg_match('/\b(security|safe|secure|protect|hack|scam|trust)\b/i', $msg)) {
            return "🔒 **Security & Safety:**\n\n**We Protect Your Account:**\n• Encrypted data transmission (SSL)\n• Secure password hashing\n• KYC verification for withdrawals\n• Admin approval for all transactions\n• Regular security audits\n\n**You Should:**\n• Use a strong, unique password\n• Never share your password\n• Enable two-factor authentication (if available)\n• Double-check wallet addresses before withdrawing\n• Be cautious of phishing attempts\n\n✅ Your funds and data are safe with us!";
        }

        // ===== TRANSACTION HISTORY =====
        if (preg_match('/\b(history|transactions|past|previous|record)\b/i', $msg)) {
            return "📜 **Transaction History:**\n\nYou can view all your past transactions in the **Wallet** page under the 'Transaction History' section.\n\n**What You'll See:**\n• Deposit records\n• Withdrawal records\n• Transaction status (pending/approved/declined)\n• Dates and amounts\n• Cryptocurrency types\n\n💡 **Tip:** Use this to track your deposits and withdrawals!";
        }

        // ===== SUPPORT/HELP QUERIES =====
        if (preg_match('/\b(support|help|assist|problem|issue|stuck)\b/i', $msg)) {
            return "🆘 **How Can I Help You?**\n\nI can assist with:\n\n✅ **Balance & Wallet** - Check your balance, view funds\n✅ **Deposits** - How to add funds\n✅ **Withdrawals** - How to cash out\n✅ **Trading** - How to place trades\n✅ **KYC Verification** - Identity verification\n✅ **Fees** - Understanding costs\n✅ **Security** - Account safety\n✅ **Transaction History** - View past activity\n\n💬 **Just ask me anything!** For example:\n• \"What's my balance?\"\n• \"How do I deposit?\"\n• \"How to withdraw funds?\"\n• \"What's my KYC status?\"\n\nIf you need human support, our team will review your message and respond shortly!";
        }

        // ===== GREETINGS =====
        if (preg_match('/\b(hello|hi|hey|greetings|good morning|good afternoon|good evening)\b/i', $msg)) {
            $greeting = "👋 Hello {$user->name}! Welcome to our support chat!\n\n";
            $greeting .= "💰 **Your Total Balance:** $" . number_format($totalBalanceUSDT, 2) . " USDT\n\n";
            $greeting .= "How can I assist you today? You can ask me about:\n";
            $greeting .= "• Your balance and wallet\n";
            $greeting .= "• Deposits and withdrawals\n";
            $greeting .= "• Trading and profits\n";
            $greeting .= "• KYC verification\n";
            $greeting .= "• Fees and security\n\n";
            $greeting .= "Just type your question! 😊";
            return $greeting;
        }

        // ===== THANK YOU =====
        if (preg_match('/\b(thank|thanks|appreciate|grateful)\b/i', $msg)) {
            return "😊 You're very welcome! I'm here to help anytime.\n\nIf you have more questions, feel free to ask. Happy trading! 🚀";
        }

        // ===== PROFIT/LOSS QUERIES =====
        if (preg_match('/\b(profit|earn|money|win|gain|loss|lose)\b/i', $msg)) {
            return "💹 **Understanding Profits & Losses:**\n\n**How Trading Works:**\n• Predict if price goes UP (Call) or DOWN (Put)\n• If you're correct, you earn profit based on leverage\n• If wrong, you lose your trade amount\n\n**Profit Calculation:**\n• Profit = Trade Amount × Leverage × Price Movement %\n• Higher leverage = Higher potential profit (and risk!)\n\n**Your Profits:**\n• Automatically added to your wallet when trade closes\n• View in Transaction History\n\n💡 **Pro Tip:** Start small, learn the market, then scale up!";
        }

        // ===== TIME/PROCESSING QUERIES =====
        if (preg_match('/\b(how long|when|time|duration|wait|pending|processing)\b/i', $msg)) {
            return "⏰ **Processing Times:**\n\n**Deposits:** Usually approved within 24 hours after verification\n**Withdrawals:** Processed within 24 hours after approval\n**KYC Verification:** 24-48 hours\n**Trades:** Instant execution, closes at expiry time\n\n💡 **Tip:** Check your transaction status in the Wallet page!";
        }

        // ===== CRYPTOCURRENCY SPECIFIC =====
        if (preg_match('/\b(bitcoin|btc|ethereum|eth|solana|sol|ripple|xrp|dogecoin|doge|usdt|usdc)\b/i', $msg)) {
            return "🪙 **Supported Cryptocurrencies:**\n\n• **BTC** (Bitcoin) - The original cryptocurrency\n• **ETH** (Ethereum) - Smart contract platform\n• **SOL** (Solana) - High-speed blockchain\n• **XRP** (Ripple) - Fast international transfers\n• **DOGE** (Dogecoin) - Popular meme coin\n• **USDT** (Tether) - Stablecoin pegged to USD\n• **USDC** (USD Coin) - Regulated stablecoin\n\n💰 **Your Balances:**\n" . implode("\n", array_map(fn($crypto, $amt) => "• {$crypto}: " . number_format($amt, 8), array_keys($balances), $balances)) . "\n\nYou can deposit, withdraw, and trade all of these!";
        }

        // ===== DEFAULT INTELLIGENT RESPONSE =====
        return "🤖 I'm here to help! I can answer questions about:\n\n💰 Balance & Wallets\n💵 Deposits & Withdrawals\n📈 Trading & Profits\n🔐 KYC & Verification\n💳 Fees & Charges\n🔒 Security\n\n**Your Current Balance:** $" . number_format($totalBalanceUSDT, 2) . " USDT\n\nTry asking:\n• \"What's my balance?\"\n• \"How do I deposit?\"\n• \"How to withdraw?\"\n• \"What's my KYC status?\"\n\nOur support team will also review your message if you need additional assistance! 😊";
    }

    // -------------------------
    // Admin Methods
    // -------------------------

    // Render Admin Support Page
    public function index()
    {
        return Inertia::render('Admin/AdminSupport', [
            'auth' => ['user' => Auth::user()],
        ]);
    }

    // API: Get all chats for Admin (with latest message)
    public function getAllChats()
    {
        // Ensure admin
        if ((int) Auth::user()->role !== 1) return response()->json(['error' => 'Unauthorized'], 403);

        $chats = Chat::with(['user', 'messages' => function($q) {
            $q->latest();
        }])->latest()->get()->map(function($chat) {
            $chat->last_message = $chat->messages->first();
            $chat->unread_count = $chat->messages->where('sender_type', 'user')->where('is_read', false)->count();
            return $chat;
        });

        return response()->json($chats);
    }

    // API: Get messages for a specific chat (Admin view)
    public function getChatMessages($chatId)
    {
        if ((int) Auth::user()->role !== 1) return response()->json(['error' => 'Unauthorized'], 403);

        $chat = Chat::findOrFail($chatId);
        
        // Mark user messages as read
        $chat->messages()->where('sender_type', 'user')->update(['is_read' => true]);

        $messages = $chat->messages()->oldest()->get();

        return response()->json($messages);
    }

    // API: Admin replies to a chat
    public function adminReply(Request $request, $chatId)
    {
        if ((int) Auth::user()->role !== 1) return response()->json(['error' => 'Unauthorized'], 403);
        $request->validate(['message' => 'required|string']);

        $chat = Chat::findOrFail($chatId);

        $msg = $chat->messages()->create([
            'sender_type' => 'admin',
            'message' => $request->message,
            'is_read' => false,
        ]);

        return redirect()->back();
    }
}
