<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\VerificationCode;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Inertia\Inertia;

class VerifyEmailCodeController extends Controller
{
    /**
     * Show the email verification form
     */
    public function show(Request $request)
    {
        return Inertia::render('Auth/VerifyEmail', [
            'email' => $request->query('email'),
            'message' => session('message'),
        ]);
    }

    /**
     * Verify the email code
     */
    public function verify(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'code' => 'required|string|size:6',
        ]);

        // Find user by email
        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return back()->withErrors(['code' => 'Invalid email address.']);
        }

        // Find the verification code
        $verificationCode = VerificationCode::where('user_id', $user->id)
            ->where('type', 'email_verification')
            ->where('code', $request->code)
            ->whereNull('verified_at')
            ->first();

        if (!$verificationCode) {
            return back()->withErrors(['code' => 'Invalid verification code.']);
        }

        if ($verificationCode->isExpired()) {
            return back()->withErrors(['code' => 'Verification code has expired. Please request a new one.']);
        }

        // Mark code as verified
        $verificationCode->markAsVerified();

        // Mark email as verified
        $user->email_verified_at = now();
        $user->save();

        // Log the user in
        Auth::login($user);

        return redirect()->route('dashboard')
            ->with('success', 'Email verified successfully! Welcome to CoinSova.');
    }

    /**
     * Resend verification code
     */
    public function resend(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return back()->withErrors(['email' => 'User not found.']);
        }

        if ($user->email_verified_at) {
            return back()->withErrors(['email' => 'Email is already verified.']);
        }

        // Generate new code
        $verificationCode = VerificationCode::createForUser($user->id, 'email_verification', 10);

        // Send email
        try {
            Mail::send('emails.verification-code', ['code' => $verificationCode->code, 'user' => $user], function($message) use ($user) {
                $message->to($user->email)
                        ->subject('Verify Your Email - CoinSova');
            });

            return back()->with('message', 'A new verification code has been sent to your email.');
        } catch (\Exception $e) {
            \Log::error('Failed to resend verification email: ' . $e->getMessage());
            return back()->withErrors(['email' => 'Failed to send email. Please try again later.']);
        }
    }
}
