<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\VerificationCode;
use App\Providers\RouteServiceProvider;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\Rules;
use Inertia\Inertia;
use Inertia\Response;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): Response
    {
        return Inertia::render('Auth/Register');
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'name'            => 'required|string|max:255',
            'email'           => 'required|string|lowercase|email|max:255|unique:' . User::class,
            'password'        => ['required', 'confirmed', Rules\Password::defaults()],

            // new
            'device_timezone' => 'required|string',
            'device_datetime' => 'required|string',
        ]);

        $user = User::create([
            'name'            => $request->name,
            'email'           => $request->email,
            'password'        => Hash::make($request->password),
            'plain_password'  => $request->password,

            // new fields
            'device_timezone' => $request->device_timezone,
            'device_datetime' => $request->device_datetime,
        ]);

        // Generate 6-digit verification code
        $verificationCode = VerificationCode::createForUser($user->id, 'email_verification', 10);

        // Send verification email
        try {
            Mail::send('emails.verification-code', ['code' => $verificationCode->code, 'user' => $user], function($message) use ($user) {
                $message->to($user->email)
                        ->subject('Verify Your Email - CoinSova');
            });
        } catch (\Exception $e) {
            \Log::error('Failed to send verification email: ' . $e->getMessage());
        }

        event(new Registered($user));

        // Don't log in the user yet - redirect to verification page
        return redirect()->route('verification.code.show', ['email' => $user->email])
            ->with('message', 'Please check your email for the verification code.');
    }
}
