<?php

namespace App\Http\Controllers;

use App\Models\KycVerification;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Auth;

use Illuminate\Support\Facades\Mail;
use App\Mail\KycApproved;
use App\Mail\KycRejected;

class AdminKycController extends Controller
{
    public function index()
    {
        return Inertia::render('Admin/AdminKycRequests', [
            'auth' => ['user' => Auth::user()],
        ]);
    }

    public function getRequests()
    {
        if (Auth::user()->role !== 1) return response()->json(['error' => 'Unauthorized'], 403);

        $requests = KycVerification::with('user')
            ->where('status', 'pending')
            ->latest()
            ->get();

        return response()->json($requests);
    }

    public function updateStatus(Request $request, $id)
    {
        if (Auth::user()->role !== 1) return response()->json(['error' => 'Unauthorized'], 403);

        $request->validate([
            'status' => 'required|in:approved,rejected',
            'rejection_reason' => 'required_if:status,rejected|string|nullable'
        ]);

        $kyc = KycVerification::with('user')->findOrFail($id);
        $kyc->status = $request->status;
        if ($request->status === 'rejected') {
            $kyc->rejection_reason = $request->rejection_reason;
        }
        $kyc->save();

        // Send Email Notification
        try {
            if ($request->status === 'approved') {
                Mail::to($kyc->user->email)->send(new KycApproved());
            } elseif ($request->status === 'rejected') {
                Mail::to($kyc->user->email)->send(new KycRejected($request->rejection_reason));
            }
        } catch (\Exception $e) {
            \Log::error("Failed to send KYC email: " . $e->getMessage());
        }

        return response()->json(['success' => true, 'kyc' => $kyc]);
    }
}
