<?php

namespace App\Http\Controllers;

use App\Models\ChartManipulation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Inertia\Inertia;

class AdminChartController extends Controller
{
    /**
     * Display the admin page for chart manipulation.
     */
    public function index()
    {
        return Inertia::render('Admin/ChartManipulator', [
            'auth' => ['user' => auth()->user()],
            'manipulations' => ChartManipulation::where('is_active', true)
                ->orderBy('timestamp', 'desc')
                ->get()
        ]);
    }

    /**
     * Store a new candle manipulation.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'symbol' => 'required|string|uppercase',
            'timestamp' => 'required|numeric',
            'open' => 'required|numeric',
            'high' => 'required|numeric',
            'low' => 'required|numeric',
            'close' => 'required|numeric',
        ]);

        ChartManipulation::updateOrCreate(
            [
                'symbol' => $validated['symbol'],
                'timestamp' => $validated['timestamp']
            ],
            [
                'open' => $validated['open'],
                'high' => $validated['high'],
                'low' => $validated['low'],
                'close' => $validated['close'],
                'is_active' => true
            ]
        );

        return redirect()->back()->with('success', 'Candle manipulation saved.');
    }

    /**
     * Get blended chart data (Real + Manipulated).
     */
    public function getChartData(Request $request)
    {
        $symbol = strtoupper($request->query('symbol', 'BTC'));
        $interval = $request->query('interval', '1m'); // Binance format: 1m, 5m, 1h, 1d

        // 1. Fetch Real Data from Binance (Public API)
        // Limit 1000 candles to give good history
        $binanceUrl = "https://api.binance.com/api/v3/klines?symbol={$symbol}USDT&interval={$interval}&limit=1000";
        
        try {
            $response = Http::timeout(10)->get($binanceUrl);
            if ($response->failed()) {
                return response()->json(['error' => 'Failed to fetch market data'], 500);
            }
            $klines = $response->json();
            
            // Check if we got valid data
            if (!is_array($klines) || empty($klines)) {
                return response()->json(['error' => 'No market data available'], 404);
            }
        } catch (\Exception $e) {
            // Fallback for local dev if internet fails or rate limit
            \Log::error('Chart data fetch error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch market data: ' . $e->getMessage()], 500);
        }

        // Format Binance Data: [time, open, high, low, close, volume, ...]
        // We need: { time: unix_seconds, open, high, low, close }
        // Binance returns millisecond timestamps, Lightweight Charts uses Seconds
        $chartData = array_map(function ($k) {
            return [
                'time' => $k[0] / 1000,
                'open' => (float)$k[1],
                'high' => (float)$k[2],
                'low' => (float)$k[3],
                'close' => (float)$k[4],
            ];
        }, $klines);

        // 2. Fetch Manipulations from DB
        $manipulations = ChartManipulation::where('symbol', $symbol)
            ->where('is_active', true)
            ->get()
            ->keyBy('timestamp'); // Key by timestamp for fast lookup

        // 3. Merge Manipulation
        // If a manipulation exists for a specific candle time, overwrite it.
        // Note: Binance returns aligned timestamps (e.g. 12:00, 12:01).
        // Ensure admin timestamps align.
        
        $mergedData = [];
        $latestRealTime = end($chartData)['time'];

        foreach ($chartData as $candle) {
            $ts = $candle['time'];
            if (isset($manipulations[$ts])) {
                // Overwrite with manipulated values
                $m = $manipulations[$ts];
                $mergedData[] = [
                    'time' => $ts,
                    'open' => (float)$m->open,
                    'high' => (float)$m->high,
                    'low' => (float)$m->low,
                    'close' => (float)$m->close,
                    'isManipulated' => true // Optional flag for frontend debugging
                ];
                // Remove from map to check for future candles later
                $manipulations->forget($ts);
            } else {
                $mergedData[] = $candle;
            }
        }

        // 4. Append Future Manipulations (that didn't match existing history)
        // Use the remaining manipulations
        foreach ($manipulations as $m) {
            if ($m->timestamp > $latestRealTime) {
                $mergedData[] = [
                    'time' => (int)$m->timestamp,
                    'open' => (float)$m->open,
                    'high' => (float)$m->high,
                    'low' => (float)$m->low,
                    'close' => (float)$m->close,
                    'isManipulated' => true
                ];
            }
        }
        
        // Sort by time just in case
        usort($mergedData, function ($a, $b) {
            return $a['time'] <=> $b['time'];
        });

        // Ensure we return at least some data
        if (empty($mergedData)) {
            return response()->json(['error' => 'No chart data available'], 404);
        }

        return response()->json($mergedData);
    }
}
