<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Trade;
use App\Models\User;
use App\Models\WalletTransaction;

class AdminTradeController extends Controller
{
    public function index()
    {
        // Load user + proper proof path + ASC order
        $trades = Trade::with('user')
            ->orderBy('id', 'DESC')
            ->get()
            ->map(function ($trade) {
                $trade->proof_url = $trade->proof
                    ? asset('storage/' . $trade->proof)
                    : null;

                return $trade;
            });

        $users = User::orderBy('name', 'ASC')->get(['id', 'name']);

        return Inertia::render('Admin/AdminTradeDashboard', [
            'trades' => $trades,
            'users' => $users,
        ]);
    }

    public function approve($id)
    {
        $trade = Trade::findOrFail($id);
        $trade->status = 'approved';
        
        // If already settled with a loss, flip it to a profit of the same amount
        if ($trade->delivery_price !== null && $trade->profit !== null && $trade->profit < 0) {
            $trade->profit = abs($trade->profit);
        }

        $trade->save();

        $this->syncTradeWalletTransaction($trade);

// Wallet transaction moved to settlement phase
        // WalletTransaction::create([
        //     'user_id' => $trade->user_id,
        //     'crypto'  => $trade->asset,
        //     'amount'  => $trade->amount,
        //     'type'    => 'withdraw',
        //     'status'  => 'approved',
        // ]);

        return back()->with('success', 'Trade approved!');
    }

    public function decline($id)
    {
        $trade = Trade::findOrFail($id);
        $trade->status = 'declined';

        // If already settled with a profit, flip it to a loss of the same amount
        if ($trade->delivery_price !== null && $trade->profit !== null && $trade->profit > 0) {
            $trade->profit = -abs($trade->profit);
        }

        $trade->save();

        $this->syncTradeWalletTransaction($trade);

        return back()->with('error', 'Trade declined!');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'asset' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
            'direction' => 'required|in:buy,sell',
            'leverage' => 'nullable|numeric|min:1',
            'status' => 'required|in:pending,approved,declined',
            'profit' => 'nullable|numeric',
            'purchase_price' => 'nullable|numeric|min:0',
            'delivery_price' => 'nullable|numeric|min:0',
        ]);

        $trade = Trade::create($validated);
        $trade->load('user');
        $trade->proof_url = $trade->proof ? asset('storage/' . $trade->proof) : null;

        return response()->json([
            'success' => true,
            'trade' => $trade,
        ]);
    }

    public function update(Request $request, $id)
    {
        $trade = Trade::findOrFail($id);

        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'asset' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
            'direction' => 'required|in:buy,sell',
            'leverage' => 'nullable|numeric|min:1',
            'status' => 'required|in:pending,approved,declined',
            'profit' => 'nullable|numeric',
            'purchase_price' => 'nullable|numeric|min:0',
            'delivery_price' => 'nullable|numeric|min:0',
        ]);

        $trade->update($validated);
        $this->syncTradeWalletTransaction($trade);
        $trade->load('user');
        $trade->proof_url = $trade->proof ? asset('storage/' . $trade->proof) : null;

        return response()->json([
            'success' => true,
            'trade' => $trade,
        ]);
    }

    public function destroy($id)
    {
        $trade = Trade::findOrFail($id);
        
        // Also delete linked transaction if it exists
        WalletTransaction::where('source', 'trade')
            ->where('source_id', $trade->id)
            ->delete();

        $trade->delete();

        return response()->json([
            'success' => true,
        ]);
    }

    /**
     * Synchronize the trade's profit/loss with a WalletTransaction.
     */
    protected function syncTradeWalletTransaction(Trade $trade)
    {
        // If the trade is NOT settled, ensure no transaction exists for it
        if ($trade->delivery_price === null || $trade->profit === null) {
            WalletTransaction::where('source', 'trade')
                ->where('source_id', $trade->id)
                ->delete();
            return;
        }

        $transaction = WalletTransaction::where('source', 'trade')
            ->where('source_id', $trade->id)
            ->first();

        // Fallback for older transactions that didn't have source_id
        if (!$transaction) {
            $transaction = WalletTransaction::where('source', 'trade')
                ->where('user_id', $trade->user_id)
                ->where('crypto', $trade->asset)
                ->where('created_at', '>=', $trade->updated_at->subMinutes(60))
                ->where('created_at', '<=', $trade->updated_at->addMinutes(60))
                ->first();
            
            if ($transaction) {
                $transaction->source_id = $trade->id;
                $transaction->save();
            }
        }

        $amount = abs($trade->profit);
        $type = $trade->profit > 0 ? 'deposit' : 'withdraw';

        if ($transaction) {
            $transaction->update([
                'amount' => $amount,
                'type' => $type,
                'status' => 'approved',
                'crypto' => $trade->asset, // Keep in sync if asset changed
                'source_id' => $trade->id,
            ]);
        } else {
            // Create the transaction if it doesn't exist yet
            WalletTransaction::create([
                'user_id' => $trade->user_id,
                'crypto' => $trade->asset,
                'amount' => $amount,
                'type' => $type,
                'status' => 'approved',
                'source' => 'trade',
                'source_id' => $trade->id,
            ]);
        }
    }
}
