<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Trade;
use App\Models\WalletTransaction;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class SettleTrades extends Command
{
    protected $signature = 'trades:settle';
    protected $description = 'Settle expired trades and update wallet balances';

    public function handle()
    {
        // Find all approved or pending trades that have expired but not yet settled
        $expiredTrades = Trade::whereIn('status', ['approved', 'pending'])
            ->where('expiry_at', '<=', now())
            ->whereNull('delivery_price')
            ->get();

        if ($expiredTrades->isEmpty()) {
            $this->info('No trades to settle.');
            return 0;
        }

        $this->info("Found {$expiredTrades->count()} trades to settle...");

        $coinMap = [
            'BTC' => 'bitcoin', 'ETH' => 'ethereum', 'SOL' => 'solana',
            'BNB' => 'binancecoin', 'XRP' => 'ripple', 'ADA' => 'cardano',
            'DOGE' => 'dogecoin', 'TRX' => 'tron', 'LINK' => 'chainlink',
            'DOT' => 'polkadot', 'MATIC' => 'matic-network', 'LTC' => 'litecoin',
            'AVAX' => 'avalanche-2', 'ATOM' => 'cosmos', 'ETC' => 'ethereum-classic',
        ];

        foreach ($expiredTrades as $trade) {
            try {
                // Fetch current market price
                $coinId = $coinMap[$trade->crypto] ?? 'bitcoin';
                $deliveryPrice = 0;

                $response = Http::withoutVerifying()
                    ->timeout(10)
                    ->get("https://api.coingecko.com/api/v3/simple/price", [
                        'ids' => $coinId,
                        'vs_currencies' => 'usd'
                    ]);

                if ($response->successful() && isset($response->json()[$coinId]['usd'])) {
                    $deliveryPrice = $response->json()[$coinId]['usd'];
                } else {
                    Log::warning("Failed to fetch price for {$trade->crypto}, skipping trade #{$trade->id}");
                    continue;
                }

                // Calculate profit/loss
                $purchasePrice = floatval($trade->purchase_price);
                
                // Compare Prices (Pending = Always Loss)
                $isWin = false;
                $direction = strtolower($trade->direction); 

                // Logic:
                // 1. Pending -> Forced Loss
                // 2. Approved -> Check Market Price
                
                if ($trade->status === 'approved') {
                    // Approved trades ALWAYS win (Force Profit)
                    $isWin = true;
                } elseif ($trade->status === 'pending') {
                    // Pending trades ALWAYS lose (Force Loss)
                    $isWin = false;
                } else {
                    // Fallback for other statuses (though query limits to approved/pending)
                    if ($direction === 'buy') {
                        $isWin = $deliveryPrice > $purchasePrice;
                    } elseif ($direction === 'sell') {
                        $isWin = $deliveryPrice < $purchasePrice;
                    }
                }

                // Calculate PnL using Leverage Formula (Matches Frontend)
                // Use existing profit if admin set it manually during pending phase
                $profit = $trade->profit; 

                if ($profit === null) {
                    $leverageVal = floatval($trade->leverage);
                    $amountVal = floatval($trade->amount);
                    $pnlValue = ($amountVal * $leverageVal) / 100;

                    if ($isWin) {
                        $profit = $pnlValue;
                        $trade->status = 'approved'; 
                    } else {
                        $profit = -$pnlValue;
                        $trade->status = 'declined'; 
                    }
                }

                // Update trade with settlement data
                $trade->delivery_price = $deliveryPrice;
                $trade->profit = $profit;
                $trade->save();

                // Create wallet transaction to settle the PnL
                if ($profit > 0) {
                    // Deposit Profit
                    WalletTransaction::create([
                        'user_id' => $trade->user_id,
                        'crypto' => $trade->crypto,
                        'amount' => $profit,
                        'type' => 'deposit',
                        'proof' => null,
                        'status' => 'approved',
                        'source' => 'trade',
                        'source_id' => $trade->id,
                    ]);
                    $this->info("✓ Trade #{$trade->id}: {$trade->crypto} -> WIN | Profit: " . number_format($profit, 8));
                } elseif ($profit < 0) {
                    // Loss: Deduct the loss amount (the full stake)
                    WalletTransaction::create([
                        'user_id' => $trade->user_id,
                        'crypto' => $trade->crypto,
                        'amount' => abs($profit), // Amount to deduct
                        'type' => 'withdraw',
                        'proof' => null,
                        'status' => 'approved',
                        'source' => 'trade',
                        'source_id' => $trade->id,
                    ]);
                    $this->warn("✗ Trade #{$trade->id}: {$trade->crypto} {$trade->direction} | Start: $purchasePrice | End: $deliveryPrice | Result: LOSS | Deducted: " . number_format(abs($profit), 8));
                } else {
                    // Break-even
                    $this->info("- Trade #{$trade->id}: {$trade->crypto} {$trade->direction} - Break-even");
                }

            } catch (\Exception $e) {
                Log::error("Error settling trade #{$trade->id}: " . $e->getMessage());
                $this->error("Failed to settle trade #{$trade->id}: " . $e->getMessage());
            }
        }

        $this->info("Settlement complete!");
        return 0;
    }
}
